"use client"

import { useState, useEffect } from 'react'
import { AdminLayout } from '@/components/admin/admin-layout'
import { Button } from '@/components/ui/button'
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card'
import { Database, Download, Upload, Trash2, RefreshCw } from 'lucide-react'
import { toast } from 'sonner'

export default function DatabasePage() {
  const [loading, setLoading] = useState(false)
  const [stats, setStats] = useState({
    totalRecords: 0,
    lastBackup: null as string | null,
    databaseSize: '0 MB'
  })

  useEffect(() => {
    fetchDatabaseStats()
  }, [])

  const fetchDatabaseStats = async () => {
    try {
      const response = await fetch('/api/admin/database', {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('adminToken') || 'admin-token-placeholder'}`
        }
      })
      if (response.ok) {
        const data = await response.json()
        setStats(data)
      }
    } catch (error) {
      console.error('Error fetching database stats:', error)
    }
  }

  const handleBackup = async () => {
    setLoading(true)
    try {
      const response = await fetch('/api/admin/database', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${localStorage.getItem('adminToken') || 'admin-token-placeholder'}`
        },
        body: JSON.stringify({ action: 'backup' })
      })
      
      if (response.ok) {
        toast.success('Database backup created successfully')
        fetchDatabaseStats()
      } else {
        toast.error('Failed to create backup')
      }
    } catch (error) {
      toast.error('Error creating backup')
    } finally {
      setLoading(false)
    }
  }

  const handleRestore = async () => {
    if (!confirm('Are you sure you want to restore the database? This will overwrite current data.')) return
    
    setLoading(true)
    try {
      const response = await fetch('/api/admin/database', {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${localStorage.getItem('adminToken') || 'admin-token-placeholder'}`
        },
        body: JSON.stringify({ action: 'restore' })
      })
      
      if (response.ok) {
        toast.success('Database restored successfully')
        fetchDatabaseStats()
      } else {
        toast.error('Failed to restore database')
      }
    } catch (error) {
      toast.error('Error restoring database')
    } finally {
      setLoading(false)
    }
  }

  return (
    <AdminLayout>
      <div className="space-y-6">
        <div className="flex items-center space-x-2">
          <Database className="w-6 h-6 text-purple-400" />
          <h1 className="text-2xl font-bold text-white">Database Management</h1>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          <Card className="bg-white/10 border-white/20">
            <CardHeader>
              <CardTitle className="text-white">Total Records</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-3xl font-bold text-purple-400">{stats.totalRecords}</p>
            </CardContent>
          </Card>

          <Card className="bg-white/10 border-white/20">
            <CardHeader>
              <CardTitle className="text-white">Database Size</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-3xl font-bold text-blue-400">{stats.databaseSize}</p>
            </CardContent>
          </Card>

          <Card className="bg-white/10 border-white/20">
            <CardHeader>
              <CardTitle className="text-white">Last Backup</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-lg text-gray-300">
                {stats.lastBackup ? new Date(stats.lastBackup).toLocaleDateString() : 'Never'}
              </p>
            </CardContent>
          </Card>
        </div>

        <Card className="bg-white/10 border-white/20">
          <CardHeader>
            <CardTitle className="text-white">Database Operations</CardTitle>
            <CardDescription className="text-gray-300">
              Manage your database backups and maintenance
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="flex flex-wrap gap-4">
              <Button 
                onClick={handleBackup} 
                disabled={loading}
                className="bg-green-600 hover:bg-green-700"
              >
                <Download className="w-4 h-4 mr-2" />
                Create Backup
              </Button>
              
              <Button 
                onClick={handleRestore} 
                disabled={loading}
                variant="outline"
                className="border-yellow-500 text-yellow-500 hover:bg-yellow-500 hover:text-black"
              >
                <Upload className="w-4 h-4 mr-2" />
                Restore Backup
              </Button>
              
              <Button 
                onClick={fetchDatabaseStats} 
                disabled={loading}
                variant="outline"
                className="border-blue-500 text-blue-500 hover:bg-blue-500 hover:text-white"
              >
                <RefreshCw className="w-4 h-4 mr-2" />
                Refresh Stats
              </Button>
            </div>
          </CardContent>
        </Card>
      </div>
    </AdminLayout>
  )
}