"use client"

import { useState, useEffect } from 'react'
import { useRouter } from 'next/navigation'
import { AdminLayout } from '@/components/admin/admin-layout'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Label } from '@/components/ui/label'
import { Textarea } from '@/components/ui/textarea'
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card'
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs'
import { Switch } from '@/components/ui/switch'
import { Mail, Send, Settings } from 'lucide-react'
import { toast } from 'sonner'

interface EmailSettings {
  smtpHost: string
  smtpPort: number
  smtpUsername: string
  smtpPassword: string
  smtpFromName: string
  smtpFromEmail: string
  smtpSecure: boolean
  emailTemplates: {
    welcome: { subject: string; body: string }
    quote: { subject: string; body: string }
    notification: { subject: string; body: string }
  }
}

export default function EmailSettingsPage() {
  const router = useRouter()
  const [settings, setSettings] = useState<EmailSettings | null>(null)
  const [loading, setLoading] = useState(true)
  const [saving, setSaving] = useState(false)
  const [testEmail, setTestEmail] = useState('')
  const [testing, setTesting] = useState(false)

  useEffect(() => {
    const isLoggedIn = localStorage.getItem("adminLoggedIn")
    if (isLoggedIn !== "true") {
      router.push("/admin")
      return
    }
    fetchEmailSettings()
  }, [router])

  const fetchEmailSettings = async () => {
    try {
      const response = await fetch('/api/admin/email', {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('adminToken') || 'admin-token-placeholder'}`
        }
      })
      if (response.ok) {
        const data = await response.json()
        setSettings(data)
      } else {
        toast.error('Failed to load email settings')
      }
    } catch (error) {
      toast.error('Error loading email settings')
    } finally {
      setLoading(false)
    }
  }

  const saveSettings = async () => {
    if (!settings) return
    
    setSaving(true)
    try {
      const response = await fetch('/api/admin/email', {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${localStorage.getItem('adminToken') || 'admin-token-placeholder'}`
        },
        body: JSON.stringify(settings)
      })
      
      if (response.ok) {
        toast.success('Email settings saved successfully')
      } else {
        toast.error('Failed to save email settings')
      }
    } catch (error) {
      toast.error('Error saving email settings')
    } finally {
      setSaving(false)
    }
  }

  const testEmailConfig = async () => {
    if (!testEmail) {
      toast.error('Please enter a test email address')
      return
    }
    
    setTesting(true)
    try {
      const response = await fetch('/api/admin/email', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${localStorage.getItem('adminToken') || 'admin-token-placeholder'}`
        },
        body: JSON.stringify({ testEmail })
      })
      
      if (response.ok) {
        toast.success('Test email sent successfully')
      } else {
        toast.error('Failed to send test email')
      }
    } catch (error) {
      toast.error('Error sending test email')
    } finally {
      setTesting(false)
    }
  }

  if (loading) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-slate-900 via-purple-900 to-slate-900 flex items-center justify-center">
        <div className="text-white text-xl">Loading...</div>
      </div>
    )
  }

  if (!settings) {
    return (
      <AdminLayout>
        <div className="text-white">Failed to load email settings</div>
      </AdminLayout>
    )
  }

  return (
    <AdminLayout>
      <div className="space-y-6">
        <div className="flex items-center space-x-2">
          <Mail className="w-6 h-6 text-purple-400" />
          <h1 className="text-2xl font-bold text-white">Email Settings</h1>
        </div>

        <Tabs defaultValue="smtp" className="space-y-4">
          <TabsList className="bg-white/10">
            <TabsTrigger value="smtp" className="text-white data-[state=active]:bg-purple-600">
              <Settings className="w-4 h-4 mr-2" />
              SMTP Configuration
            </TabsTrigger>
            <TabsTrigger value="templates" className="text-white data-[state=active]:bg-purple-600">
              <Mail className="w-4 h-4 mr-2" />
              Email Templates
            </TabsTrigger>
            <TabsTrigger value="test" className="text-white data-[state=active]:bg-purple-600">
              <Send className="w-4 h-4 mr-2" />
              Test Email
            </TabsTrigger>
          </TabsList>

          <TabsContent value="smtp">
            <Card className="bg-white/10 border-white/20">
              <CardHeader>
                <CardTitle className="text-white">SMTP Configuration</CardTitle>
                <CardDescription className="text-gray-300">
                  Configure your email server settings
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="smtpHost" className="text-white">SMTP Host</Label>
                    <Input
                      id="smtpHost"
                      value={settings.smtpHost}
                      onChange={(e) => setSettings({...settings, smtpHost: e.target.value})}
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>
                  <div>
                    <Label htmlFor="smtpPort" className="text-white">SMTP Port</Label>
                    <Input
                      id="smtpPort"
                      type="number"
                      value={settings.smtpPort}
                      onChange={(e) => setSettings({...settings, smtpPort: parseInt(e.target.value)})}
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>
                </div>
                
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="smtpUsername" className="text-white">Username</Label>
                    <Input
                      id="smtpUsername"
                      value={settings.smtpUsername}
                      onChange={(e) => setSettings({...settings, smtpUsername: e.target.value})}
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>
                  <div>
                    <Label htmlFor="smtpPassword" className="text-white">Password</Label>
                    <Input
                      id="smtpPassword"
                      type="password"
                      value={settings.smtpPassword}
                      onChange={(e) => setSettings({...settings, smtpPassword: e.target.value})}
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>
                </div>
                
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="smtpFromName" className="text-white">From Name</Label>
                    <Input
                      id="smtpFromName"
                      value={settings.smtpFromName}
                      onChange={(e) => setSettings({...settings, smtpFromName: e.target.value})}
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>
                  <div>
                    <Label htmlFor="smtpFromEmail" className="text-white">From Email</Label>
                    <Input
                      id="smtpFromEmail"
                      type="email"
                      value={settings.smtpFromEmail}
                      onChange={(e) => setSettings({...settings, smtpFromEmail: e.target.value})}
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>
                </div>
                
                <div className="flex items-center space-x-2">
                  <Switch
                    id="smtpSecure"
                    checked={settings.smtpSecure}
                    onCheckedChange={(checked) => setSettings({...settings, smtpSecure: checked})}
                  />
                  <Label htmlFor="smtpSecure" className="text-white">Use SSL/TLS</Label>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="templates">
            <div className="space-y-4">
              {Object.entries(settings.emailTemplates).map(([key, template]) => (
                <Card key={key} className="bg-white/10 border-white/20">
                  <CardHeader>
                    <CardTitle className="text-white capitalize">{key} Email Template</CardTitle>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    <div>
                      <Label className="text-white">Subject</Label>
                      <Input
                        value={template.subject}
                        onChange={(e) => setSettings({
                          ...settings,
                          emailTemplates: {
                            ...settings.emailTemplates,
                            [key]: { ...template, subject: e.target.value }
                          }
                        })}
                        className="bg-white/10 border-white/20 text-white"
                      />
                    </div>
                    <div>
                      <Label className="text-white">Body</Label>
                      <Textarea
                        value={template.body}
                        onChange={(e) => setSettings({
                          ...settings,
                          emailTemplates: {
                            ...settings.emailTemplates,
                            [key]: { ...template, body: e.target.value }
                          }
                        })}
                        className="bg-white/10 border-white/20 text-white min-h-[100px]"
                      />
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </TabsContent>

          <TabsContent value="test">
            <Card className="bg-white/10 border-white/20">
              <CardHeader>
                <CardTitle className="text-white">Test Email Configuration</CardTitle>
                <CardDescription className="text-gray-300">
                  Send a test email to verify your SMTP settings
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label htmlFor="testEmail" className="text-white">Test Email Address</Label>
                  <Input
                    id="testEmail"
                    type="email"
                    value={testEmail}
                    onChange={(e) => setTestEmail(e.target.value)}
                    placeholder="Enter email address to test"
                    className="bg-white/10 border-white/20 text-white"
                  />
                </div>
                <Button
                  onClick={testEmailConfig}
                  disabled={testing || !testEmail}
                  className="bg-purple-600 hover:bg-purple-700"
                >
                  {testing ? (
                    <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                  ) : (
                    <Send className="w-4 h-4 mr-2" />
                  )}
                  Send Test Email
                </Button>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>

        <div className="flex justify-end">
          <Button
            onClick={saveSettings}
            disabled={saving}
            className="bg-green-600 hover:bg-green-700"
          >
            {saving ? (
              <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
            ) : null}
            Save Settings
          </Button>
        </div>
      </div>
    </AdminLayout>
  )
}