"use client"

import { useState, useEffect } from "react"
import { useRouter } from "next/navigation"
import { AdminLayout } from "@/components/admin/admin-layout"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Badge } from "@/components/ui/badge"
import { Progress } from "@/components/ui/progress"
import {
  Monitor,
  Activity,
  Database,
  Users,
  AlertTriangle,
  CheckCircle,
  XCircle,
  Server,
  HardDrive,
  Cpu,
  MemoryStick,
  Globe,
  Shield,
  Download,
  RefreshCw,
  Play,
  Pause,
  Square,
  Trash2,
} from "lucide-react"

interface SystemMetrics {
  cpu: number
  memory: number
  disk: number
  network: number
  uptime: string
  activeUsers: number
  totalRequests: number
  errorRate: number
}

interface LogEntry {
  id: string
  timestamp: string
  level: "info" | "warning" | "error" | "critical"
  message: string
  source: string
}

interface Service {
  id: string
  name: string
  status: "running" | "stopped" | "error"
  uptime: string
  memory: number
  cpu: number
}

export default function SystemMonitoring() {
  const router = useRouter()
  const [loading, setLoading] = useState(true)
  const [refreshing, setRefreshing] = useState(false)
  const [autoRefresh, setAutoRefresh] = useState(true)

  const [metrics, setMetrics] = useState<SystemMetrics>({
    cpu: 45,
    memory: 68,
    disk: 32,
    network: 12,
    uptime: "7 days, 14 hours",
    activeUsers: 234,
    totalRequests: 15847,
    errorRate: 0.2,
  })

  const [logs, setLogs] = useState<LogEntry[]>([
    {
      id: "1",
      timestamp: "2024-01-20 14:30:25",
      level: "info",
      message: "User login successful",
      source: "Authentication",
    },
    {
      id: "2",
      timestamp: "2024-01-20 14:28:15",
      level: "warning",
      message: "High memory usage detected",
      source: "System Monitor",
    },
    {
      id: "3",
      timestamp: "2024-01-20 14:25:10",
      level: "error",
      message: "Database connection timeout",
      source: "Database",
    },
    {
      id: "4",
      timestamp: "2024-01-20 14:20:05",
      level: "info",
      message: "Backup completed successfully",
      source: "Backup Service",
    },
    {
      id: "5",
      timestamp: "2024-01-20 14:15:30",
      level: "critical",
      message: "Security alert: Multiple failed login attempts",
      source: "Security",
    },
  ])

  const [services, setServices] = useState<Service[]>([
    {
      id: "web",
      name: "Web Server",
      status: "running",
      uptime: "7d 14h 30m",
      memory: 512,
      cpu: 15,
    },
    {
      id: "db",
      name: "Database",
      status: "running",
      uptime: "7d 14h 30m",
      memory: 1024,
      cpu: 25,
    },
    {
      id: "cache",
      name: "Cache Service",
      status: "running",
      uptime: "7d 14h 30m",
      memory: 256,
      cpu: 5,
    },
    {
      id: "email",
      name: "Email Service",
      status: "stopped",
      uptime: "0m",
      memory: 0,
      cpu: 0,
    },
    {
      id: "backup",
      name: "Backup Service",
      status: "running",
      uptime: "2h 15m",
      memory: 128,
      cpu: 8,
    },
  ])

  useEffect(() => {
    const isLoggedIn = localStorage.getItem("adminLoggedIn")
    if (isLoggedIn !== "true") {
      router.push("/admin")
      return
    }
    setLoading(false)

    // Auto-refresh metrics every 30 seconds
    const interval = setInterval(() => {
      if (autoRefresh) {
        refreshMetrics()
      }
    }, 30000)

    return () => clearInterval(interval)
  }, [router, autoRefresh])

  const refreshMetrics = async () => {
    setRefreshing(true)

    // Simulate API call to get real metrics
    await new Promise((resolve) => setTimeout(resolve, 1000))

    setMetrics((prev) => ({
      ...prev,
      cpu: Math.max(10, Math.min(90, prev.cpu + (Math.random() - 0.5) * 20)),
      memory: Math.max(20, Math.min(95, prev.memory + (Math.random() - 0.5) * 15)),
      disk: Math.max(10, Math.min(80, prev.disk + (Math.random() - 0.5) * 10)),
      network: Math.max(0, Math.min(50, prev.network + (Math.random() - 0.5) * 25)),
      activeUsers: Math.max(100, prev.activeUsers + Math.floor((Math.random() - 0.5) * 50)),
      totalRequests: prev.totalRequests + Math.floor(Math.random() * 100),
    }))

    setRefreshing(false)
  }

  const handleServiceAction = (serviceId: string, action: "start" | "stop" | "restart") => {
    setServices((prev) =>
      prev.map((service) => {
        if (service.id === serviceId) {
          switch (action) {
            case "start":
              return { ...service, status: "running" as const }
            case "stop":
              return { ...service, status: "stopped" as const, memory: 0, cpu: 0 }
            case "restart":
              return { ...service, status: "running" as const }
            default:
              return service
          }
        }
        return service
      }),
    )
  }

  const getStatusColor = (status: string) => {
    switch (status) {
      case "running":
        return "text-green-400"
      case "stopped":
        return "text-red-400"
      case "error":
        return "text-orange-400"
      default:
        return "text-gray-400"
    }
  }

  const getLogLevelColor = (level: string) => {
    switch (level) {
      case "info":
        return "text-blue-400"
      case "warning":
        return "text-yellow-400"
      case "error":
        return "text-orange-400"
      case "critical":
        return "text-red-400"
      default:
        return "text-gray-400"
    }
  }

  if (loading) {
    return (
      <AdminLayout>
        <div className="flex items-center justify-center h-64">
          <div className="text-white text-xl">Loading system monitor...</div>
        </div>
      </AdminLayout>
    )
  }

  return (
    <AdminLayout>
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-white">System Monitoring & Controls</h1>
            <p className="text-gray-300">Monitor system performance and manage services</p>
          </div>

          <div className="flex items-center space-x-2">
            <Button
              variant="outline"
              onClick={() => setAutoRefresh(!autoRefresh)}
              className={`border-gray-500/30 hover:bg-gray-500/10 bg-transparent ${
                autoRefresh ? "text-green-400" : "text-gray-400"
              }`}
            >
              {autoRefresh ? <Pause className="w-4 h-4 mr-2" /> : <Play className="w-4 h-4 mr-2" />}
              Auto Refresh
            </Button>

            <Button
              onClick={refreshMetrics}
              disabled={refreshing}
              className="bg-gradient-to-r from-purple-500 to-blue-500 hover:from-purple-600 hover:to-blue-600"
            >
              {refreshing ? (
                <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
              ) : (
                <RefreshCw className="w-4 h-4 mr-2" />
              )}
              Refresh
            </Button>
          </div>
        </div>

        {/* System Overview */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          <Card className="bg-white/10 backdrop-blur-lg border-white/20">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-300 text-sm">CPU Usage</p>
                  <p className="text-2xl font-bold text-white">{metrics.cpu}%</p>
                  <Progress value={metrics.cpu} className="mt-2" />
                </div>
                <Cpu className="w-8 h-8 text-purple-400" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-white/10 backdrop-blur-lg border-white/20">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-300 text-sm">Memory Usage</p>
                  <p className="text-2xl font-bold text-white">{metrics.memory}%</p>
                  <Progress value={metrics.memory} className="mt-2" />
                </div>
                <MemoryStick className="w-8 h-8 text-blue-400" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-white/10 backdrop-blur-lg border-white/20">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-300 text-sm">Disk Usage</p>
                  <p className="text-2xl font-bold text-white">{metrics.disk}%</p>
                  <Progress value={metrics.disk} className="mt-2" />
                </div>
                <HardDrive className="w-8 h-8 text-green-400" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-white/10 backdrop-blur-lg border-white/20">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-300 text-sm">Network I/O</p>
                  <p className="text-2xl font-bold text-white">{metrics.network} MB/s</p>
                  <Progress value={metrics.network * 2} className="mt-2" />
                </div>
                <Globe className="w-8 h-8 text-orange-400" />
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Additional Stats */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
          <Card className="bg-white/10 backdrop-blur-lg border-white/20">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-300 text-sm">System Uptime</p>
                  <p className="text-xl font-bold text-white">{metrics.uptime}</p>
                </div>
                <Activity className="w-6 h-6 text-green-400" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-white/10 backdrop-blur-lg border-white/20">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-300 text-sm">Active Users</p>
                  <p className="text-xl font-bold text-white">{metrics.activeUsers}</p>
                </div>
                <Users className="w-6 h-6 text-blue-400" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-white/10 backdrop-blur-lg border-white/20">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-300 text-sm">Total Requests</p>
                  <p className="text-xl font-bold text-white">{metrics.totalRequests.toLocaleString()}</p>
                </div>
                <Monitor className="w-6 h-6 text-purple-400" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-white/10 backdrop-blur-lg border-white/20">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-300 text-sm">Error Rate</p>
                  <p className="text-xl font-bold text-white">{metrics.errorRate}%</p>
                </div>
                <AlertTriangle className="w-6 h-6 text-orange-400" />
              </div>
            </CardContent>
          </Card>
        </div>

        <Tabs defaultValue="services" className="space-y-6">
          <TabsList className="bg-white/10 border-white/20">
            <TabsTrigger value="services" className="data-[state=active]:bg-purple-500/20">
              <Server className="w-4 h-4 mr-2" />
              Services
            </TabsTrigger>
            <TabsTrigger value="logs" className="data-[state=active]:bg-purple-500/20">
              <Activity className="w-4 h-4 mr-2" />
              System Logs
            </TabsTrigger>
            <TabsTrigger value="database" className="data-[state=active]:bg-purple-500/20">
              <Database className="w-4 h-4 mr-2" />
              Database
            </TabsTrigger>
            <TabsTrigger value="security" className="data-[state=active]:bg-purple-500/20">
              <Shield className="w-4 h-4 mr-2" />
              Security
            </TabsTrigger>
          </TabsList>

          <TabsContent value="services" className="space-y-6">
            <Card className="bg-white/10 backdrop-blur-lg border-white/20">
              <CardHeader>
                <CardTitle className="text-white">System Services</CardTitle>
                <CardDescription className="text-gray-300">Monitor and control system services</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {services.map((service) => (
                    <div
                      key={service.id}
                      className="flex items-center justify-between p-4 bg-white/5 rounded-lg border border-white/10"
                    >
                      <div className="flex items-center space-x-4">
                        <div className="flex items-center space-x-2">
                          {service.status === "running" ? (
                            <CheckCircle className="w-5 h-5 text-green-400" />
                          ) : service.status === "error" ? (
                            <XCircle className="w-5 h-5 text-red-400" />
                          ) : (
                            <XCircle className="w-5 h-5 text-gray-400" />
                          )}
                          <span className="text-white font-medium">{service.name}</span>
                        </div>
                        <Badge
                          variant={
                            service.status === "running"
                              ? "default"
                              : service.status === "error"
                                ? "destructive"
                                : "secondary"
                          }
                        >
                          {service.status}
                        </Badge>
                      </div>

                      <div className="flex items-center space-x-6">
                        <div className="text-right">
                          <p className="text-white text-sm">Uptime: {service.uptime}</p>
                          <p className="text-gray-400 text-xs">
                            CPU: {service.cpu}% | Memory: {service.memory}MB
                          </p>
                        </div>

                        <div className="flex items-center space-x-2">
                          <Button
                            size="sm"
                            variant="outline"
                            className="border-green-500/30 text-green-400 hover:bg-green-500/10 bg-transparent"
                            onClick={() => handleServiceAction(service.id, "start")}
                            disabled={service.status === "running"}
                          >
                            <Play className="w-3 h-3" />
                          </Button>
                          <Button
                            size="sm"
                            variant="outline"
                            className="border-red-500/30 text-red-400 hover:bg-red-500/10 bg-transparent"
                            onClick={() => handleServiceAction(service.id, "stop")}
                            disabled={service.status === "stopped"}
                          >
                            <Square className="w-3 h-3" />
                          </Button>
                          <Button
                            size="sm"
                            variant="outline"
                            className="border-blue-500/30 text-blue-400 hover:bg-blue-500/10 bg-transparent"
                            onClick={() => handleServiceAction(service.id, "restart")}
                          >
                            <RefreshCw className="w-3 h-3" />
                          </Button>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="logs" className="space-y-6">
            <Card className="bg-white/10 backdrop-blur-lg border-white/20">
              <CardHeader>
                <div className="flex items-center justify-between">
                  <div>
                    <CardTitle className="text-white">System Logs</CardTitle>
                    <CardDescription className="text-gray-300">Recent system events and messages</CardDescription>
                  </div>
                  <div className="flex items-center space-x-2">
                    <Button
                      size="sm"
                      variant="outline"
                      className="border-blue-500/30 text-blue-400 hover:bg-blue-500/10 bg-transparent"
                    >
                      <Download className="w-4 h-4 mr-2" />
                      Export
                    </Button>
                    <Button
                      size="sm"
                      variant="outline"
                      className="border-red-500/30 text-red-400 hover:bg-red-500/10 bg-transparent"
                    >
                      <Trash2 className="w-4 h-4 mr-2" />
                      Clear
                    </Button>
                  </div>
                </div>
              </CardHeader>
              <CardContent>
                <div className="space-y-2 max-h-96 overflow-y-auto">
                  {logs.map((log) => (
                    <div
                      key={log.id}
                      className="flex items-start space-x-4 p-3 bg-white/5 rounded-lg border border-white/10"
                    >
                      <div className="flex-shrink-0">
                        <Badge
                          variant={
                            log.level === "critical" || log.level === "error"
                              ? "destructive"
                              : log.level === "warning"
                                ? "secondary"
                                : "default"
                          }
                        >
                          {log.level}
                        </Badge>
                      </div>
                      <div className="flex-1 min-w-0">
                        <p className="text-white text-sm">{log.message}</p>
                        <p className="text-gray-400 text-xs">
                          {log.timestamp} • {log.source}
                        </p>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="database" className="space-y-6">
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
              <Card className="bg-white/10 backdrop-blur-lg border-white/20">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-gray-300 text-sm">Active Connections</p>
                      <p className="text-2xl font-bold text-white">24</p>
                    </div>
                    <Database className="w-8 h-8 text-blue-400" />
                  </div>
                </CardContent>
              </Card>

              <Card className="bg-white/10 backdrop-blur-lg border-white/20">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-gray-300 text-sm">Query Performance</p>
                      <p className="text-2xl font-bold text-white">45ms</p>
                    </div>
                    <Activity className="w-8 h-8 text-green-400" />
                  </div>
                </CardContent>
              </Card>

              <Card className="bg-white/10 backdrop-blur-lg border-white/20">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-gray-300 text-sm">Database Size</p>
                      <p className="text-2xl font-bold text-white">2.4 GB</p>
                    </div>
                    <HardDrive className="w-8 h-8 text-purple-400" />
                  </div>
                </CardContent>
              </Card>
            </div>

            <Card className="bg-white/10 backdrop-blur-lg border-white/20">
              <CardHeader>
                <CardTitle className="text-white">Database Operations</CardTitle>
                <CardDescription className="text-gray-300">Manage database backups and maintenance</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="flex flex-wrap gap-4">
                  <Button className="bg-gradient-to-r from-green-500 to-emerald-500 hover:from-green-600 hover:to-emerald-600">
                    <Download className="w-4 h-4 mr-2" />
                    Create Backup
                  </Button>
                  <Button
                    variant="outline"
                    className="border-blue-500/30 text-blue-400 hover:bg-blue-500/10 bg-transparent"
                  >
                    <RefreshCw className="w-4 h-4 mr-2" />
                    Optimize Tables
                  </Button>
                  <Button
                    variant="outline"
                    className="border-purple-500/30 text-purple-400 hover:bg-purple-500/10 bg-transparent"
                  >
                    <Activity className="w-4 h-4 mr-2" />
                    Run Diagnostics
                  </Button>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="security" className="space-y-6">
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
              <Card className="bg-white/10 backdrop-blur-lg border-white/20">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-gray-300 text-sm">Failed Logins</p>
                      <p className="text-2xl font-bold text-white">12</p>
                    </div>
                    <AlertTriangle className="w-8 h-8 text-red-400" />
                  </div>
                </CardContent>
              </Card>

              <Card className="bg-white/10 backdrop-blur-lg border-white/20">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-gray-300 text-sm">Blocked IPs</p>
                      <p className="text-2xl font-bold text-white">3</p>
                    </div>
                    <Shield className="w-8 h-8 text-orange-400" />
                  </div>
                </CardContent>
              </Card>

              <Card className="bg-white/10 backdrop-blur-lg border-white/20">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-gray-300 text-sm">Security Score</p>
                      <p className="text-2xl font-bold text-white">98%</p>
                    </div>
                    <CheckCircle className="w-8 h-8 text-green-400" />
                  </div>
                </CardContent>
              </Card>
            </div>

            <Card className="bg-white/10 backdrop-blur-lg border-white/20">
              <CardHeader>
                <CardTitle className="text-white">Security Actions</CardTitle>
                <CardDescription className="text-gray-300">Manage security settings and monitoring</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="flex flex-wrap gap-4">
                  <Button className="bg-gradient-to-r from-red-500 to-pink-500 hover:from-red-600 hover:to-pink-600">
                    <Shield className="w-4 h-4 mr-2" />
                    Security Scan
                  </Button>
                  <Button
                    variant="outline"
                    className="border-orange-500/30 text-orange-400 hover:bg-orange-500/10 bg-transparent"
                  >
                    <AlertTriangle className="w-4 h-4 mr-2" />
                    View Threats
                  </Button>
                  <Button
                    variant="outline"
                    className="border-blue-500/30 text-blue-400 hover:bg-blue-500/10 bg-transparent"
                  >
                    <Download className="w-4 h-4 mr-2" />
                    Export Logs
                  </Button>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </AdminLayout>
  )
}
