"use client"

import type React from "react"

import { useState, useEffect } from "react"
import { useRouter } from "next/navigation"
import { AdminLayout } from "@/components/admin/admin-layout"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Textarea } from "@/components/ui/textarea"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Switch } from "@/components/ui/switch"
import {
  Settings,
  Shield,
  Mail,
  CreditCard,
  Bell,
  Globe,
  Key,
  Save,
  RefreshCw,
  Download,
  Upload,
  Eye,
  EyeOff,
} from "lucide-react"

export default function SettingsPanel() {
  const router = useRouter()
  const [loading, setLoading] = useState(true)
  const [saving, setSaving] = useState(false)
  const [showPasswords, setShowPasswords] = useState(false)

  const [settings, setSettings] = useState({
    admin: {
      username: "Admin",
      password: "Admin",
      email: "admin@bishopknight.com",
      twoFactorEnabled: false,
    },
    email: {
      smtpHost: "smtp.gmail.com",
      smtpPort: "587",
      smtpUsername: "your-email@gmail.com",
      smtpPassword: "your-app-password",
      imapHost: "imap.gmail.com",
      imapPort: "993",
      popHost: "pop.gmail.com",
      popPort: "995",
      fromName: "Bishop & Knight Insurance",
      fromEmail: "noreply@bishopknight.com",
    },
    payment: {
      bankName: "First Bank of Nigeria",
      accountName: "Bishop & Knight Insurance Brokers Limited",
      accountNumber: "1234567890",
      sortCode: "011-152-003",
      paymentGateway: "paystack",
      paystackPublicKey: "pk_test_xxxxx",
      paystackSecretKey: "sk_test_xxxxx",
      flutterwavePublicKey: "FLWPUBK_TEST-xxxxx",
      flutterwaveSecretKey: "FLWSECK_TEST-xxxxx",
    },
    social: {
      facebook: "https://facebook.com/bishopknight",
      twitter: "https://twitter.com/bishopknight",
      linkedin: "https://linkedin.com/company/bishopknight",
      instagram: "https://instagram.com/bishopknight",
      youtube: "https://youtube.com/@bishopknight",
    },
    notifications: {
      emailNotifications: true,
      smsNotifications: false,
      pushNotifications: true,
      newQuoteAlert: true,
      paymentAlert: true,
      systemAlert: true,
    },
    site: {
      siteName: "Bishop & Knight Insurance Brokers",
      siteUrl: "https://bishopknight.com",
      timezone: "Africa/Lagos",
      language: "en",
      maintenanceMode: false,
      registrationEnabled: true,
      quotesEnabled: true,
    },
    security: {
      sessionTimeout: "30",
      maxLoginAttempts: "5",
      passwordMinLength: "8",
      requireStrongPassword: true,
      enableCaptcha: false,
      ipWhitelist: "",
    },
  })

  useEffect(() => {
    const isLoggedIn = localStorage.getItem("adminLoggedIn")
    if (isLoggedIn !== "true") {
      router.push("/admin")
      return
    }

    // Load settings from localStorage
    const savedSettings = localStorage.getItem("adminSettings")
    if (savedSettings) {
      setSettings(JSON.parse(savedSettings))
    }
    setLoading(false)
  }, [router])

  const handleSave = async () => {
    setSaving(true)

    // Update admin credentials in localStorage
    localStorage.setItem("adminUsername", settings.admin.username)
    localStorage.setItem("adminPassword", settings.admin.password)

    // Save all settings
    localStorage.setItem("adminSettings", JSON.stringify(settings))

    // Simulate API delay
    await new Promise((resolve) => setTimeout(resolve, 1000))
    setSaving(false)
  }

  const handleExportSettings = () => {
    const dataStr = JSON.stringify(settings, null, 2)
    const dataBlob = new Blob([dataStr], { type: "application/json" })
    const url = URL.createObjectURL(dataBlob)
    const link = document.createElement("a")
    link.href = url
    link.download = "admin-settings-backup.json"
    link.click()
  }

  const handleImportSettings = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0]
    if (file) {
      const reader = new FileReader()
      reader.onload = (e) => {
        try {
          const importedSettings = JSON.parse(e.target?.result as string)
          setSettings(importedSettings)
        } catch (error) {
          alert("Invalid settings file")
        }
      }
      reader.readAsText(file)
    }
  }

  if (loading) {
    return (
      <AdminLayout>
        <div className="flex items-center justify-center h-64">
          <div className="text-white text-xl">Loading settings...</div>
        </div>
      </AdminLayout>
    )
  }

  return (
    <AdminLayout>
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-white">Settings & Configuration</h1>
            <p className="text-gray-300">Manage system settings and configurations</p>
          </div>

          <div className="flex items-center space-x-2">
            <Button
              variant="outline"
              onClick={handleExportSettings}
              className="border-blue-500/30 text-blue-400 hover:bg-blue-500/10 bg-transparent"
            >
              <Download className="w-4 h-4 mr-2" />
              Export
            </Button>

            <label className="cursor-pointer">
              <Button
                variant="outline"
                className="border-green-500/30 text-green-400 hover:bg-green-500/10 bg-transparent"
                asChild
              >
                <span>
                  <Upload className="w-4 h-4 mr-2" />
                  Import
                </span>
              </Button>
              <input type="file" accept=".json" onChange={handleImportSettings} className="hidden" />
            </label>

            <Button
              onClick={handleSave}
              disabled={saving}
              className="bg-gradient-to-r from-purple-500 to-blue-500 hover:from-purple-600 hover:to-blue-600"
            >
              {saving ? (
                <>
                  <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                  Saving...
                </>
              ) : (
                <>
                  <Save className="w-4 h-4 mr-2" />
                  Save All
                </>
              )}
            </Button>
          </div>
        </div>

        <Tabs defaultValue="admin" className="space-y-6">
          <TabsList className="bg-white/10 border-white/20 flex-wrap">
            <TabsTrigger value="admin" className="data-[state=active]:bg-purple-500/20">
              <Shield className="w-4 h-4 mr-2" />
              Admin
            </TabsTrigger>
            <TabsTrigger value="email" className="data-[state=active]:bg-purple-500/20">
              <Mail className="w-4 h-4 mr-2" />
              Email
            </TabsTrigger>
            <TabsTrigger value="payment" className="data-[state=active]:bg-purple-500/20">
              <CreditCard className="w-4 h-4 mr-2" />
              Payment
            </TabsTrigger>
            <TabsTrigger value="social" className="data-[state=active]:bg-purple-500/20">
              <Globe className="w-4 h-4 mr-2" />
              Social
            </TabsTrigger>
            <TabsTrigger value="notifications" className="data-[state=active]:bg-purple-500/20">
              <Bell className="w-4 h-4 mr-2" />
              Notifications
            </TabsTrigger>
            <TabsTrigger value="site" className="data-[state=active]:bg-purple-500/20">
              <Settings className="w-4 h-4 mr-2" />
              Site
            </TabsTrigger>
            <TabsTrigger value="security" className="data-[state=active]:bg-purple-500/20">
              <Key className="w-4 h-4 mr-2" />
              Security
            </TabsTrigger>
          </TabsList>

          <TabsContent value="admin" className="space-y-6">
            <Card className="bg-white/10 backdrop-blur-lg border-white/20">
              <CardHeader>
                <CardTitle className="text-white">Admin Account Settings</CardTitle>
                <CardDescription className="text-gray-300">
                  Manage admin credentials and account settings
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label className="text-white">Username</Label>
                    <Input
                      value={settings.admin.username}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          admin: { ...settings.admin, username: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">Password</Label>
                    <div className="relative">
                      <Input
                        type={showPasswords ? "text" : "password"}
                        value={settings.admin.password}
                        onChange={(e) =>
                          setSettings({
                            ...settings,
                            admin: { ...settings.admin, password: e.target.value },
                          })
                        }
                        className="bg-white/10 border-white/20 text-white pr-10"
                      />
                      <Button
                        type="button"
                        variant="ghost"
                        size="sm"
                        className="absolute right-0 top-0 h-full px-3 text-gray-400 hover:text-white"
                        onClick={() => setShowPasswords(!showPasswords)}
                      >
                        {showPasswords ? <EyeOff className="w-4 h-4" /> : <Eye className="w-4 h-4" />}
                      </Button>
                    </div>
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">Email</Label>
                    <Input
                      type="email"
                      value={settings.admin.email}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          admin: { ...settings.admin, email: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">Two-Factor Authentication</Label>
                    <div className="flex items-center space-x-2">
                      <Switch
                        checked={settings.admin.twoFactorEnabled}
                        onCheckedChange={(checked) =>
                          setSettings({
                            ...settings,
                            admin: { ...settings.admin, twoFactorEnabled: checked },
                          })
                        }
                      />
                      <span className="text-gray-300">Enable 2FA</span>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="email" className="space-y-6">
            <Card className="bg-white/10 backdrop-blur-lg border-white/20">
              <CardHeader>
                <CardTitle className="text-white">SMTP Configuration</CardTitle>
                <CardDescription className="text-gray-300">Configure outgoing email settings</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label className="text-white">SMTP Host</Label>
                    <Input
                      value={settings.email.smtpHost}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          email: { ...settings.email, smtpHost: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">SMTP Port</Label>
                    <Input
                      value={settings.email.smtpPort}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          email: { ...settings.email, smtpPort: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">SMTP Username</Label>
                    <Input
                      value={settings.email.smtpUsername}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          email: { ...settings.email, smtpUsername: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">SMTP Password</Label>
                    <Input
                      type={showPasswords ? "text" : "password"}
                      value={settings.email.smtpPassword}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          email: { ...settings.email, smtpPassword: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">From Name</Label>
                    <Input
                      value={settings.email.fromName}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          email: { ...settings.email, fromName: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">From Email</Label>
                    <Input
                      type="email"
                      value={settings.email.fromEmail}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          email: { ...settings.email, fromEmail: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card className="bg-white/10 backdrop-blur-lg border-white/20">
              <CardHeader>
                <CardTitle className="text-white">IMAP/POP Configuration</CardTitle>
                <CardDescription className="text-gray-300">Configure incoming email settings</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  <div className="space-y-2">
                    <Label className="text-white">IMAP Host</Label>
                    <Input
                      value={settings.email.imapHost}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          email: { ...settings.email, imapHost: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">IMAP Port</Label>
                    <Input
                      value={settings.email.imapPort}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          email: { ...settings.email, imapPort: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">POP Host</Label>
                    <Input
                      value={settings.email.popHost}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          email: { ...settings.email, popHost: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="payment" className="space-y-6">
            <Card className="bg-white/10 backdrop-blur-lg border-white/20">
              <CardHeader>
                <CardTitle className="text-white">Bank Account Details</CardTitle>
                <CardDescription className="text-gray-300">Configure bank transfer information</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label className="text-white">Bank Name</Label>
                    <Input
                      value={settings.payment.bankName}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          payment: { ...settings.payment, bankName: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">Account Name</Label>
                    <Input
                      value={settings.payment.accountName}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          payment: { ...settings.payment, accountName: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">Account Number</Label>
                    <Input
                      value={settings.payment.accountNumber}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          payment: { ...settings.payment, accountNumber: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">Sort Code</Label>
                    <Input
                      value={settings.payment.sortCode}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          payment: { ...settings.payment, sortCode: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card className="bg-white/10 backdrop-blur-lg border-white/20">
              <CardHeader>
                <CardTitle className="text-white">Payment Gateway Settings</CardTitle>
                <CardDescription className="text-gray-300">Configure online payment processors</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-2">
                  <Label className="text-white">Primary Gateway</Label>
                  <Select
                    value={settings.payment.paymentGateway}
                    onValueChange={(value) =>
                      setSettings({
                        ...settings,
                        payment: { ...settings.payment, paymentGateway: value },
                      })
                    }
                  >
                    <SelectTrigger className="bg-white/10 border-white/20 text-white">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="paystack">Paystack</SelectItem>
                      <SelectItem value="flutterwave">Flutterwave</SelectItem>
                      <SelectItem value="interswitch">Interswitch</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label className="text-white">Paystack Public Key</Label>
                    <Input
                      value={settings.payment.paystackPublicKey}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          payment: { ...settings.payment, paystackPublicKey: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">Paystack Secret Key</Label>
                    <Input
                      type={showPasswords ? "text" : "password"}
                      value={settings.payment.paystackSecretKey}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          payment: { ...settings.payment, paystackSecretKey: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="social" className="space-y-6">
            <Card className="bg-white/10 backdrop-blur-lg border-white/20">
              <CardHeader>
                <CardTitle className="text-white">Social Media Links</CardTitle>
                <CardDescription className="text-gray-300">Configure social media profiles</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-4">
                  <div className="space-y-2">
                    <Label className="text-white">Facebook</Label>
                    <Input
                      value={settings.social.facebook}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          social: { ...settings.social, facebook: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                      placeholder="https://facebook.com/yourpage"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">Twitter</Label>
                    <Input
                      value={settings.social.twitter}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          social: { ...settings.social, twitter: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                      placeholder="https://twitter.com/yourhandle"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">LinkedIn</Label>
                    <Input
                      value={settings.social.linkedin}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          social: { ...settings.social, linkedin: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                      placeholder="https://linkedin.com/company/yourcompany"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">Instagram</Label>
                    <Input
                      value={settings.social.instagram}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          social: { ...settings.social, instagram: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                      placeholder="https://instagram.com/yourhandle"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">YouTube</Label>
                    <Input
                      value={settings.social.youtube}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          social: { ...settings.social, youtube: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                      placeholder="https://youtube.com/@yourchannel"
                    />
                  </div>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="notifications" className="space-y-6">
            <Card className="bg-white/10 backdrop-blur-lg border-white/20">
              <CardHeader>
                <CardTitle className="text-white">Notification Settings</CardTitle>
                <CardDescription className="text-gray-300">Configure system notifications and alerts</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-4">
                  <div className="flex items-center justify-between">
                    <div>
                      <Label className="text-white">Email Notifications</Label>
                      <p className="text-gray-400 text-sm">Receive notifications via email</p>
                    </div>
                    <Switch
                      checked={settings.notifications.emailNotifications}
                      onCheckedChange={(checked) =>
                        setSettings({
                          ...settings,
                          notifications: { ...settings.notifications, emailNotifications: checked },
                        })
                      }
                    />
                  </div>

                  <div className="flex items-center justify-between">
                    <div>
                      <Label className="text-white">SMS Notifications</Label>
                      <p className="text-gray-400 text-sm">Receive notifications via SMS</p>
                    </div>
                    <Switch
                      checked={settings.notifications.smsNotifications}
                      onCheckedChange={(checked) =>
                        setSettings({
                          ...settings,
                          notifications: { ...settings.notifications, smsNotifications: checked },
                        })
                      }
                    />
                  </div>

                  <div className="flex items-center justify-between">
                    <div>
                      <Label className="text-white">New Quote Alerts</Label>
                      <p className="text-gray-400 text-sm">Get notified when new quotes are submitted</p>
                    </div>
                    <Switch
                      checked={settings.notifications.newQuoteAlert}
                      onCheckedChange={(checked) =>
                        setSettings({
                          ...settings,
                          notifications: { ...settings.notifications, newQuoteAlert: checked },
                        })
                      }
                    />
                  </div>

                  <div className="flex items-center justify-between">
                    <div>
                      <Label className="text-white">Payment Alerts</Label>
                      <p className="text-gray-400 text-sm">Get notified about payment transactions</p>
                    </div>
                    <Switch
                      checked={settings.notifications.paymentAlert}
                      onCheckedChange={(checked) =>
                        setSettings({
                          ...settings,
                          notifications: { ...settings.notifications, paymentAlert: checked },
                        })
                      }
                    />
                  </div>

                  <div className="flex items-center justify-between">
                    <div>
                      <Label className="text-white">System Alerts</Label>
                      <p className="text-gray-400 text-sm">Get notified about system issues and updates</p>
                    </div>
                    <Switch
                      checked={settings.notifications.systemAlert}
                      onCheckedChange={(checked) =>
                        setSettings({
                          ...settings,
                          notifications: { ...settings.notifications, systemAlert: checked },
                        })
                      }
                    />
                  </div>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="site" className="space-y-6">
            <Card className="bg-white/10 backdrop-blur-lg border-white/20">
              <CardHeader>
                <CardTitle className="text-white">Site Configuration</CardTitle>
                <CardDescription className="text-gray-300">General site settings and preferences</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label className="text-white">Site Name</Label>
                    <Input
                      value={settings.site.siteName}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          site: { ...settings.site, siteName: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">Site URL</Label>
                    <Input
                      value={settings.site.siteUrl}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          site: { ...settings.site, siteUrl: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">Timezone</Label>
                    <Select
                      value={settings.site.timezone}
                      onValueChange={(value) =>
                        setSettings({
                          ...settings,
                          site: { ...settings.site, timezone: value },
                        })
                      }
                    >
                      <SelectTrigger className="bg-white/10 border-white/20 text-white">
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="Africa/Lagos">Africa/Lagos (WAT)</SelectItem>
                        <SelectItem value="UTC">UTC</SelectItem>
                        <SelectItem value="America/New_York">America/New_York (EST)</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">Language</Label>
                    <Select
                      value={settings.site.language}
                      onValueChange={(value) =>
                        setSettings({
                          ...settings,
                          site: { ...settings.site, language: value },
                        })
                      }
                    >
                      <SelectTrigger className="bg-white/10 border-white/20 text-white">
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="en">English</SelectItem>
                        <SelectItem value="yo">Yoruba</SelectItem>
                        <SelectItem value="ig">Igbo</SelectItem>
                        <SelectItem value="ha">Hausa</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                </div>

                <div className="space-y-4">
                  <div className="flex items-center justify-between">
                    <div>
                      <Label className="text-white">Maintenance Mode</Label>
                      <p className="text-gray-400 text-sm">Put site in maintenance mode</p>
                    </div>
                    <Switch
                      checked={settings.site.maintenanceMode}
                      onCheckedChange={(checked) =>
                        setSettings({
                          ...settings,
                          site: { ...settings.site, maintenanceMode: checked },
                        })
                      }
                    />
                  </div>

                  <div className="flex items-center justify-between">
                    <div>
                      <Label className="text-white">User Registration</Label>
                      <p className="text-gray-400 text-sm">Allow new user registrations</p>
                    </div>
                    <Switch
                      checked={settings.site.registrationEnabled}
                      onCheckedChange={(checked) =>
                        setSettings({
                          ...settings,
                          site: { ...settings.site, registrationEnabled: checked },
                        })
                      }
                    />
                  </div>

                  <div className="flex items-center justify-between">
                    <div>
                      <Label className="text-white">Quote System</Label>
                      <p className="text-gray-400 text-sm">Enable quote request system</p>
                    </div>
                    <Switch
                      checked={settings.site.quotesEnabled}
                      onCheckedChange={(checked) =>
                        setSettings({
                          ...settings,
                          site: { ...settings.site, quotesEnabled: checked },
                        })
                      }
                    />
                  </div>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="security" className="space-y-6">
            <Card className="bg-white/10 backdrop-blur-lg border-white/20">
              <CardHeader>
                <CardTitle className="text-white">Security Settings</CardTitle>
                <CardDescription className="text-gray-300">Configure security and access controls</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label className="text-white">Session Timeout (minutes)</Label>
                    <Input
                      type="number"
                      value={settings.security.sessionTimeout}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          security: { ...settings.security, sessionTimeout: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">Max Login Attempts</Label>
                    <Input
                      type="number"
                      value={settings.security.maxLoginAttempts}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          security: { ...settings.security, maxLoginAttempts: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">Password Min Length</Label>
                    <Input
                      type="number"
                      value={settings.security.passwordMinLength}
                      onChange={(e) =>
                        setSettings({
                          ...settings,
                          security: { ...settings.security, passwordMinLength: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>
                </div>

                <div className="space-y-4">
                  <div className="flex items-center justify-between">
                    <div>
                      <Label className="text-white">Require Strong Password</Label>
                      <p className="text-gray-400 text-sm">Enforce strong password requirements</p>
                    </div>
                    <Switch
                      checked={settings.security.requireStrongPassword}
                      onCheckedChange={(checked) =>
                        setSettings({
                          ...settings,
                          security: { ...settings.security, requireStrongPassword: checked },
                        })
                      }
                    />
                  </div>

                  <div className="flex items-center justify-between">
                    <div>
                      <Label className="text-white">Enable CAPTCHA</Label>
                      <p className="text-gray-400 text-sm">Add CAPTCHA to forms</p>
                    </div>
                    <Switch
                      checked={settings.security.enableCaptcha}
                      onCheckedChange={(checked) =>
                        setSettings({
                          ...settings,
                          security: { ...settings.security, enableCaptcha: checked },
                        })
                      }
                    />
                  </div>
                </div>

                <div className="space-y-2">
                  <Label className="text-white">IP Whitelist</Label>
                  <Textarea
                    value={settings.security.ipWhitelist}
                    onChange={(e) =>
                      setSettings({
                        ...settings,
                        security: { ...settings.security, ipWhitelist: e.target.value },
                      })
                    }
                    className="bg-white/10 border-white/20 text-white"
                    placeholder="Enter IP addresses, one per line"
                    rows={3}
                  />
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </AdminLayout>
  )
}
