import { NextRequest, NextResponse } from 'next/server'
import { z } from 'zod'

// Payment settings schema
const PaymentSettingsSchema = z.object({
  stripePublicKey: z.string().min(1),
  stripeSecretKey: z.string().min(1),
  paypalClientId: z.string().optional(),
  paypalClientSecret: z.string().optional(),
  flutterwavePublicKey: z.string().optional(),
  flutterwaveSecretKey: z.string().optional(),
  paystackPublicKey: z.string().optional(),
  paystackSecretKey: z.string().optional(),
  currency: z.string().default('NGN'),
  testMode: z.boolean().default(true)
})

// GET - Fetch payment settings
export async function GET(request: NextRequest) {
  try {
    // Verify admin authentication
    const isAuthenticated = await verifyAdminAuth(request)
    if (!isAuthenticated) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
    }

    // Mock payment settings (replace with actual database call)
    const paymentSettings = {
      id: '1',
      stripePublicKey: process.env.STRIPE_PUBLIC_KEY || '',
      stripeSecretKey: process.env.STRIPE_SECRET_KEY || '',
      paypalClientId: process.env.PAYPAL_CLIENT_ID || '',
      paypalClientSecret: process.env.PAYPAL_CLIENT_SECRET || '',
      flutterwavePublicKey: process.env.FLUTTERWAVE_PUBLIC_KEY || '',
      flutterwaveSecretKey: process.env.FLUTTERWAVE_SECRET_KEY || '',
      paystackPublicKey: process.env.PAYSTACK_PUBLIC_KEY || '',
      paystackSecretKey: process.env.PAYSTACK_SECRET_KEY || '',
      currency: 'NGN',
      testMode: true,
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString()
    }

    return NextResponse.json(paymentSettings)
  } catch (error) {
    console.error('Payment settings fetch error:', error)
    return NextResponse.json({ error: 'Internal server error' }, { status: 500 })
  }
}

// PUT - Update payment settings
export async function PUT(request: NextRequest) {
  try {
    const body = await request.json()
    
    // Verify admin authentication
    const isAuthenticated = await verifyAdminAuth(request)
    if (!isAuthenticated) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
    }

    // Validate the request body
    const validatedSettings = PaymentSettingsSchema.parse(body)

    // Mock update (replace with actual database update)
    const updatedSettings = {
      id: '1',
      ...validatedSettings,
      updatedAt: new Date().toISOString()
    }

    return NextResponse.json(updatedSettings)
  } catch (error) {
    if (error instanceof z.ZodError) {
      return NextResponse.json({ error: 'Validation failed', details: error.errors }, { status: 400 })
    }
    console.error('Payment settings update error:', error)
    return NextResponse.json({ error: 'Internal server error' }, { status: 500 })
  }
}

// Helper function for admin authentication
async function verifyAdminAuth(request: NextRequest): Promise<boolean> {
  try {
    const authHeader = request.headers.get('authorization')
    if (!authHeader || !authHeader.startsWith('Bearer ')) {
      return false
    }

    const token = authHeader.substring(7)
    
    // Simple token validation (replace with proper JWT verification)
    if (token === process.env.ADMIN_TOKEN || token === 'admin-token-placeholder') {
      return true
    }

    return false
  } catch (error) {
    console.error('Auth verification error:', error)
    return false
  }
}