"use client"

import type React from "react"

import { useState } from "react"
import { useRouter } from "next/navigation"
import { Button } from "@/components/ui/button"
import {
  LayoutDashboard,
  FileText,
  Users,
  Settings,
  ImageIcon,
  Mail,
  ShoppingCart,
  CreditCard,
  LogOut,
  Menu,
  X,
  Shield,
  Database,
  Monitor,
} from "lucide-react"

interface AdminLayoutProps {
  children: React.ReactNode
}

export function AdminLayout({ children }: AdminLayoutProps) {  // ✅ Named export
  const [sidebarOpen, setSidebarOpen] = useState(false)
  const router = useRouter()

  const menuItems = [
    { icon: LayoutDashboard, label: "Dashboard", href: "/admin/dashboard" },
    { icon: FileText, label: "Content Management", href: "/admin/content" },
    { icon: Users, label: "User Management", href: "/admin/users" },
    { icon: ImageIcon, label: "Media Manager", href: "/admin/media" },
    { icon: ShoppingCart, label: "Shop Management", href: "/admin/shop" },
    { icon: CreditCard, label: "Payment Settings", href: "/admin/payments" },
    { icon: Mail, label: "Email Settings", href: "/admin/email" },
    { icon: Database, label: "Database", href: "/admin/database" },
    { icon: Monitor, label: "System Monitor", href: "/admin/monitor" },
    { icon: Settings, label: "Settings", href: "/admin/settings" },
  ]

  const handleLogout = () => {
    localStorage.removeItem("adminLoggedIn")
    localStorage.removeItem("adminLoginTime")
    router.push("/admin")
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-900 via-purple-900 to-slate-900">
      {sidebarOpen && (
        <div className="fixed inset-0 bg-black/50 z-40 lg:hidden" onClick={() => setSidebarOpen(false)} />
      )}

      <div
        className={`fixed inset-y-0 left-0 z-50 w-64 bg-white/10 backdrop-blur-lg border-r border-white/20 transform transition-transform duration-300 ease-in-out ${
          sidebarOpen ? "translate-x-0" : "-translate-x-full"
        } lg:translate-x-0`}
      >
        <div className="flex items-center justify-between p-4 border-b border-white/20">
          <div className="flex items-center space-x-2">
            <Shield className="w-8 h-8 text-purple-400" />
            <span className="text-white font-bold">Admin Portal</span>
          </div>
          <Button variant="ghost" size="sm" className="lg:hidden text-white" onClick={() => setSidebarOpen(false)}>
            <X className="w-4 h-4" />
          </Button>
        </div>

        <nav className="p-4 space-y-2">
          {menuItems.map((item) => (
            <Button
              key={item.href}
              variant="ghost"
              className="w-full justify-start text-white hover:bg-white/10"
              onClick={() => router.push(item.href)}
            >
              <item.icon className="w-4 h-4 mr-3" />
              {item.label}
            </Button>
          ))}
        </nav>

        <div className="absolute bottom-4 left-4 right-4">
          <Button
            variant="outline"
            className="w-full border-red-500/30 text-red-400 hover:bg-red-500/10 bg-transparent"
            onClick={handleLogout}
          >
            <LogOut className="w-4 h-4 mr-2" />
            Logout
          </Button>
        </div>
      </div>

      <div className="lg:ml-64">
        <header className="bg-white/10 backdrop-blur-lg border-b border-white/20 p-4">
          <div className="flex items-center justify-between">
            <Button variant="ghost" size="sm" className="lg:hidden text-white" onClick={() => setSidebarOpen(true)}>
              <Menu className="w-4 h-4" />
            </Button>

            <div className="flex items-center space-x-4">
              <span className="text-white text-sm">Welcome, Admin</span>
            </div>
          </div>
        </header>

        <main className="p-6">{children}</main>
      </div>
    </div>
  )
}

// Remove the default export
