"use client"

import { Cookie, Settings, BarChart3, Target, Shield, Trash2, Phone, Mail, Calendar } from "lucide-react"
import { Button } from "@/components/ui/button"

export default function CookiePolicyContent() {
  const lastUpdated = "January 15, 2025"

  const cookieTypes = [
    {
      id: "essential",
      title: "Essential Cookies",
      icon: Shield,
      description: "These cookies are necessary for the website to function properly and cannot be disabled.",
      examples: [
        "Session management cookies",
        "Security cookies",
        "Load balancing cookies",
        "Cookie consent preferences",
      ],
      retention: "Session or up to 1 year",
      canDisable: false,
    },
    {
      id: "analytics",
      title: "Analytics Cookies",
      icon: BarChart3,
      description:
        "These cookies help us understand how visitors interact with our website by collecting anonymous information.",
      examples: ["Google Analytics cookies", "Page view tracking", "User behavior analysis", "Performance monitoring"],
      retention: "Up to 2 years",
      canDisable: true,
    },
    {
      id: "marketing",
      title: "Marketing Cookies",
      icon: Target,
      description:
        "These cookies are used to deliver relevant advertisements and track the effectiveness of marketing campaigns.",
      examples: [
        "Social media tracking pixels",
        "Advertising network cookies",
        "Retargeting cookies",
        "Campaign performance tracking",
      ],
      retention: "Up to 1 year",
      canDisable: true,
    },
    {
      id: "functional",
      title: "Functional Cookies",
      icon: Settings,
      description:
        "These cookies enable enhanced functionality and personalization, such as remembering your preferences.",
      examples: ["Language preferences", "Theme settings", "Form data retention", "Chat widget functionality"],
      retention: "Up to 1 year",
      canDisable: true,
    },
  ]

  const thirdPartyServices = [
    {
      name: "Google Analytics",
      purpose: "Website analytics and performance monitoring",
      cookies: "_ga, _gid, _gat",
      privacy: "https://policies.google.com/privacy",
    },
    {
      name: "Google Ads",
      purpose: "Advertising and remarketing",
      cookies: "_gcl_au, _gcl_aw",
      privacy: "https://policies.google.com/privacy",
    },
    {
      name: "Facebook Pixel",
      purpose: "Social media advertising and analytics",
      cookies: "_fbp, _fbc",
      privacy: "https://www.facebook.com/privacy/policy",
    },
    {
      name: "LinkedIn Insight Tag",
      purpose: "Professional network advertising",
      cookies: "li_sugr, lidc, bcookie",
      privacy: "https://www.linkedin.com/legal/privacy-policy",
    },
  ]

  return (
    <div className="py-20">
      <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <div className="text-center mb-16">
          <div className="inline-flex items-center gap-2 bg-orange-100 rounded-full px-4 py-2 mb-6">
            <Cookie className="w-5 h-5 text-orange-600" />
            <span className="text-orange-800 font-medium">Cookie Usage</span>
          </div>

          <h1 className="text-4xl md:text-5xl font-bold mb-6">
            <span className="bg-gradient-to-r from-orange-600 to-red-600 bg-clip-text text-transparent">
              Cookie Policy
            </span>
          </h1>

          <p className="text-xl text-gray-600 max-w-3xl mx-auto mb-8">
            This Cookie Policy explains how Bishop & Knight Insurance Brokers Limited uses cookies and similar
            technologies to enhance your browsing experience.
          </p>

          <div className="bg-gradient-to-r from-orange-50 to-red-50 p-6 rounded-2xl border border-orange-200">
            <div className="flex items-center justify-center gap-2 text-orange-800">
              <Calendar className="w-5 h-5" />
              <span className="font-medium">Last Updated: {lastUpdated}</span>
            </div>
          </div>
        </div>

        {/* What Are Cookies */}
        <div className="bg-white rounded-3xl shadow-lg p-8 mb-12">
          <h2 className="text-2xl font-bold text-gray-900 mb-6 flex items-center">
            <Cookie className="w-6 h-6 text-orange-600 mr-3" />
            What Are Cookies?
          </h2>
          <div className="prose prose-lg text-gray-600 space-y-4">
            <p>
              Cookies are small text files that are stored on your device (computer, tablet, or mobile) when you visit a
              website. They are widely used to make websites work more efficiently and to provide information to website
              owners.
            </p>
            <p>
              Cookies help us understand how you use our website, remember your preferences, and improve your overall
              experience. They can also help us provide you with relevant content and advertisements.
            </p>
            <p>
              We use both first-party cookies (set by our website) and third-party cookies (set by other services we use
              to enhance our website functionality).
            </p>
          </div>
        </div>

        {/* Types of Cookies */}
        <div className="space-y-8 mb-12">
          <div className="text-center">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">Types of Cookies We Use</h2>
            <p className="text-lg text-gray-600">
              We use different types of cookies for various purposes to enhance your experience on our website.
            </p>
          </div>

          {cookieTypes.map((type, index) => {
            const IconComponent = type.icon
            return (
              <div key={type.id} className="bg-white rounded-3xl shadow-lg overflow-hidden">
                <div className="bg-gradient-to-r from-orange-500 to-red-500 p-6">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-3">
                      <div className="bg-white/20 rounded-full p-3">
                        <IconComponent className="w-6 h-6 text-white" />
                      </div>
                      <h3 className="text-2xl font-bold text-white">{type.title}</h3>
                    </div>
                    <div className="text-right">
                      <div
                        className={`px-3 py-1 rounded-full text-sm font-medium ${
                          type.canDisable ? "bg-yellow-100 text-yellow-800" : "bg-red-100 text-red-800"
                        }`}
                      >
                        {type.canDisable ? "Optional" : "Required"}
                      </div>
                    </div>
                  </div>
                </div>

                <div className="p-8">
                  <p className="text-gray-600 text-lg mb-6">{type.description}</p>

                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                      <h4 className="text-lg font-semibold text-gray-900 mb-3">Examples:</h4>
                      <ul className="space-y-2">
                        {type.examples.map((example, idx) => (
                          <li key={idx} className="flex items-center text-gray-600">
                            <div className="w-2 h-2 bg-orange-400 rounded-full mr-3"></div>
                            {example}
                          </li>
                        ))}
                      </ul>
                    </div>
                    <div>
                      <h4 className="text-lg font-semibold text-gray-900 mb-3">Retention Period:</h4>
                      <p className="text-gray-600">{type.retention}</p>
                      <h4 className="text-lg font-semibold text-gray-900 mb-3 mt-4">Can be disabled:</h4>
                      <p className="text-gray-600">{type.canDisable ? "Yes" : "No"}</p>
                    </div>
                  </div>
                </div>
              </div>
            )
          })}
        </div>

        {/* Third-Party Services */}
        <div className="bg-white rounded-3xl shadow-lg p-8 mb-12">
          <h2 className="text-2xl font-bold text-gray-900 mb-6">Third-Party Services</h2>
          <p className="text-gray-600 mb-8">
            We use various third-party services that may set their own cookies. Below are the main services we use and
            links to their privacy policies:
          </p>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            {thirdPartyServices.map((service, index) => (
              <div key={index} className="border border-gray-200 rounded-xl p-6">
                <h3 className="text-lg font-semibold text-gray-900 mb-2">{service.name}</h3>
                <p className="text-gray-600 text-sm mb-3">{service.purpose}</p>
                <div className="mb-3">
                  <span className="text-sm font-medium text-gray-700">Cookies: </span>
                  <span className="text-sm text-gray-600">{service.cookies}</span>
                </div>
                <a
                  href={service.privacy}
                  target="_blank"
                  rel="noopener noreferrer"
                  className="text-orange-600 hover:text-orange-700 text-sm font-medium hover:underline"
                >
                  View Privacy Policy →
                </a>
              </div>
            ))}
          </div>
        </div>

        {/* Managing Cookies */}
        <div className="bg-white rounded-3xl shadow-lg p-8 mb-12">
          <h2 className="text-2xl font-bold text-gray-900 mb-6 flex items-center">
            <Settings className="w-6 h-6 text-orange-600 mr-3" />
            Managing Your Cookie Preferences
          </h2>

          <div className="space-y-6">
            <div>
              <h3 className="text-lg font-semibold text-gray-900 mb-3">Cookie Consent Banner</h3>
              <p className="text-gray-600 mb-4">
                When you first visit our website, you'll see a cookie consent banner where you can choose which types of
                cookies to accept. You can change your preferences at any time by clicking the cookie settings link in
                our website footer.
              </p>
              <Button className="bg-gradient-to-r from-orange-600 to-red-600 hover:from-orange-700 hover:to-red-700 text-white">
                <Settings className="w-4 h-4 mr-2" />
                Manage Cookie Preferences
              </Button>
            </div>

            <div>
              <h3 className="text-lg font-semibold text-gray-900 mb-3">Browser Settings</h3>
              <p className="text-gray-600 mb-4">
                You can also control cookies through your browser settings. Most browsers allow you to:
              </p>
              <ul className="space-y-2 text-gray-600">
                <li className="flex items-center">
                  <div className="w-2 h-2 bg-orange-400 rounded-full mr-3"></div>
                  View and delete cookies
                </li>
                <li className="flex items-center">
                  <div className="w-2 h-2 bg-orange-400 rounded-full mr-3"></div>
                  Block cookies from specific websites
                </li>
                <li className="flex items-center">
                  <div className="w-2 h-2 bg-orange-400 rounded-full mr-3"></div>
                  Block third-party cookies
                </li>
                <li className="flex items-center">
                  <div className="w-2 h-2 bg-orange-400 rounded-full mr-3"></div>
                  Clear all cookies when you close your browser
                </li>
              </ul>
            </div>

            <div>
              <h3 className="text-lg font-semibold text-gray-900 mb-3">Impact of Disabling Cookies</h3>
              <p className="text-gray-600">
                Please note that disabling certain cookies may affect the functionality of our website. Essential
                cookies cannot be disabled as they are necessary for the website to function properly.
              </p>
            </div>
          </div>
        </div>

        {/* Opt-Out Links */}
        <div className="bg-gradient-to-r from-orange-50 to-red-50 rounded-3xl p-8 mb-12 border border-orange-200">
          <h2 className="text-2xl font-bold text-gray-900 mb-6 flex items-center">
            <Trash2 className="w-6 h-6 text-orange-600 mr-3" />
            Opt-Out of Tracking
          </h2>
          <p className="text-gray-600 mb-6">
            You can opt out of tracking by various advertising networks and analytics services:
          </p>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <a
              href="https://tools.google.com/dlpage/gaoptout"
              target="_blank"
              rel="noopener noreferrer"
              className="flex items-center justify-between p-4 bg-white rounded-xl border border-orange-200 hover:border-orange-300 transition-colors"
            >
              <span className="font-medium text-gray-900">Google Analytics Opt-out</span>
              <span className="text-orange-600">→</span>
            </a>
            <a
              href="https://www.facebook.com/settings?tab=ads"
              target="_blank"
              rel="noopener noreferrer"
              className="flex items-center justify-between p-4 bg-white rounded-xl border border-orange-200 hover:border-orange-300 transition-colors"
            >
              <span className="font-medium text-gray-900">Facebook Ad Preferences</span>
              <span className="text-orange-600">→</span>
            </a>
            <a
              href="https://optout.networkadvertising.org/"
              target="_blank"
              rel="noopener noreferrer"
              className="flex items-center justify-between p-4 bg-white rounded-xl border border-orange-200 hover:border-orange-300 transition-colors"
            >
              <span className="font-medium text-gray-900">NAI Opt-out Tool</span>
              <span className="text-orange-600">→</span>
            </a>
            <a
              href="https://www.youronlinechoices.com/"
              target="_blank"
              rel="noopener noreferrer"
              className="flex items-center justify-between p-4 bg-white rounded-xl border border-orange-200 hover:border-orange-300 transition-colors"
            >
              <span className="font-medium text-gray-900">Your Online Choices</span>
              <span className="text-orange-600">→</span>
            </a>
          </div>
        </div>

        {/* Contact Information */}
        <div className="bg-gradient-to-r from-orange-600 to-red-600 rounded-3xl shadow-lg p-8 mt-12 text-white">
          <h2 className="text-2xl font-bold mb-6 flex items-center">
            <Phone className="w-6 h-6 mr-3" />
            Questions About Cookies?
          </h2>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div>
              <h3 className="text-lg font-semibold mb-3">Privacy Team</h3>
              <div className="space-y-2">
                <div className="flex items-center gap-2">
                  <Mail className="w-4 h-4" />
                  <a href="mailto:privacy@bishopknight.com" className="hover:underline">
                    privacy@bishopknight.com
                  </a>
                </div>
                <div className="flex items-center gap-2">
                  <Phone className="w-4 h-4" />
                  <a href="tel:+2341234567890" className="hover:underline">
                    +234 123 456 7890
                  </a>
                </div>
              </div>
            </div>
            <div>
              <h3 className="text-lg font-semibold mb-3">Office Address</h3>
              <p className="text-orange-100">
                Bishop & Knight Insurance Brokers Limited
                <br />
                123 Victoria Island
                <br />
                Lagos, Nigeria
              </p>
            </div>
          </div>
          <div className="mt-6 p-4 bg-white/10 rounded-xl">
            <p className="text-sm text-orange-100">
              If you have any questions about our use of cookies or this Cookie Policy, please contact us using the
              information above.
            </p>
          </div>
        </div>

        {/* Updates to Policy */}
        <div className="bg-white rounded-3xl shadow-lg p-8 mt-12">
          <h2 className="text-2xl font-bold text-gray-900 mb-6">Updates to This Cookie Policy</h2>
          <div className="prose prose-lg text-gray-600 space-y-4">
            <p>
              We may update this Cookie Policy from time to time to reflect changes in our practices, technology, or
              legal requirements. We will notify you of any material changes by posting the updated policy on our
              website and updating the "Last Updated" date.
            </p>
            <p>
              We encourage you to review this Cookie Policy periodically to stay informed about how we use cookies and
              similar technologies.
            </p>
          </div>
        </div>
      </div>
    </div>
  )
}
