"use client"

import { useState } from "react"
import { Car, Heart, Home, Ship, Briefcase, Users, Shield, Building, Plus, Check } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { useShoppingCart } from "@/components/shopping-cart-provider"

const services = [
  {
    id: "motor-insurance",
    name: "Motor Insurance",
    description:
      "Comprehensive coverage for your vehicles including third-party, comprehensive, and commercial vehicle insurance.",
    price: 45000,
    originalPrice: 60000,
    icon: Car,
    category: "Personal",
    features: ["Third-party coverage", "Comprehensive protection", "Accident benefits", "24/7 roadside assistance"],
    popular: true,
    image: "https://images.unsplash.com/photo-1449824913935-59a10b8d2000?w=400&h=300&fit=crop",
  },
  {
    id: "health-insurance",
    name: "Health & Life Insurance",
    description: "Complete health and life coverage including medical expenses, hospitalization, and life protection.",
    price: 85000,
    originalPrice: 100000,
    icon: Heart,
    category: "Personal",
    features: ["Medical coverage", "Hospitalization", "Life protection", "Family coverage"],
    popular: true,
    image: "https://images.unsplash.com/photo-1559757148-5c350d0d3c56?w=400&h=300&fit=crop",
  },
  {
    id: "property-insurance",
    name: "Property Insurance",
    description: "Protect your residential and commercial properties against fire, theft, and natural disasters.",
    price: 120000,
    originalPrice: 150000,
    icon: Home,
    category: "Property",
    features: ["Fire protection", "Theft coverage", "Natural disaster", "Contents insurance"],
    popular: false,
    image: "https://images.unsplash.com/photo-1560518883-ce09059eeffa?w=400&h=300&fit=crop",
  },
  {
    id: "marine-aviation",
    name: "Marine & Aviation",
    description: "Specialized coverage for marine vessels, cargo, and aviation risks with international standards.",
    price: 200000,
    originalPrice: 250000,
    icon: Ship,
    category: "Specialized",
    features: ["Vessel coverage", "Cargo protection", "Aviation risks", "International standards"],
    popular: false,
    image: "https://images.unsplash.com/photo-1544551763-46a013bb70d5?w=400&h=300&fit=crop",
  },
  {
    id: "professional-indemnity",
    name: "Professional Indemnity",
    description:
      "Protection for professionals against claims of negligence, errors, and omissions in service delivery.",
    price: 95000,
    originalPrice: 120000,
    icon: Briefcase,
    category: "Business",
    features: ["Negligence coverage", "Error protection", "Legal defense", "Professional liability"],
    popular: false,
    image: "https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=400&h=300&fit=crop",
  },
  {
    id: "group-benefits",
    name: "Group Benefits",
    description: "Comprehensive employee benefit packages including group life, health, and pension schemes.",
    price: 150000,
    originalPrice: 180000,
    icon: Users,
    category: "Business",
    features: ["Group life insurance", "Employee health", "Pension schemes", "Disability coverage"],
    popular: true,
    image: "https://images.unsplash.com/photo-1522202176988-66273c2fd55f?w=400&h=300&fit=crop",
  },
  {
    id: "business-insurance",
    name: "Business Insurance",
    description:
      "Complete business protection including general liability, business interruption, and equipment coverage.",
    price: 175000,
    originalPrice: 220000,
    icon: Building,
    category: "Business",
    features: ["General liability", "Business interruption", "Equipment coverage", "Cyber protection"],
    popular: false,
    image: "https://images.unsplash.com/photo-1486406146926-c627a92ad1ab?w=400&h=300&fit=crop",
  },
  {
    id: "risk-advisory",
    name: "Risk Advisory Services",
    description: "Professional risk assessment and management consulting to identify and mitigate business risks.",
    price: 80000,
    originalPrice: 100000,
    icon: Shield,
    category: "Consulting",
    features: ["Risk assessment", "Management consulting", "Compliance review", "Strategic planning"],
    popular: false,
    image: "https://images.unsplash.com/photo-1454165804606-c3d57bc86b40?w=400&h=300&fit=crop",
  },
]

const categories = ["All", "Personal", "Property", "Business", "Specialized", "Consulting"]

export default function ServicesShopSection() {
  const [selectedCategory, setSelectedCategory] = useState("All")
  const { addToCart, cartItems } = useShoppingCart()

  const filteredServices =
    selectedCategory === "All" ? services : services.filter((service) => service.category === selectedCategory)

  const formatPrice = (price: number) => {
    return new Intl.NumberFormat("en-NG", {
      style: "currency",
      currency: "NGN",
      minimumFractionDigits: 0,
    }).format(price)
  }

  const isInCart = (serviceId: string) => {
    return cartItems.some((item) => item.id === serviceId)
  }

  return (
    <section id="services-shop" className="py-20 bg-gradient-to-br from-slate-50 to-blue-50">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <div className="text-center mb-16">
          <h2 className="text-4xl md:text-5xl font-bold mb-6">
            <span className="bg-gradient-to-r from-purple-600 to-blue-600 bg-clip-text text-transparent">
              Insurance Services
            </span>
          </h2>
          <p className="text-xl text-gray-600 max-w-3xl mx-auto">
            Choose from our comprehensive range of insurance products designed to protect what matters most to you.
          </p>
        </div>

        {/* Category Filter */}
        <div className="flex flex-wrap justify-center gap-3 mb-12">
          {categories.map((category) => (
            <Button
              key={category}
              variant={selectedCategory === category ? "default" : "outline"}
              onClick={() => setSelectedCategory(category)}
              className={`px-6 py-2 rounded-full transition-all duration-300 ${
                selectedCategory === category
                  ? "bg-gradient-to-r from-purple-600 to-blue-600 text-white shadow-lg"
                  : "border-purple-200 text-purple-600 hover:bg-purple-50"
              }`}
            >
              {category}
            </Button>
          ))}
        </div>

        {/* Services Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-8">
          {filteredServices.map((service) => {
            const IconComponent = service.icon
            return (
              <div
                key={service.id}
                className="group relative bg-white rounded-3xl shadow-lg hover:shadow-2xl transition-all duration-500 overflow-hidden border border-gray-100 hover:border-purple-200 transform hover:-translate-y-2"
              >
                {/* Popular Badge */}
                {service.popular && (
                  <div className="absolute top-4 right-4 z-10">
                    <Badge className="bg-gradient-to-r from-orange-500 to-pink-500 text-white px-3 py-1 rounded-full text-xs font-semibold">
                      Popular
                    </Badge>
                  </div>
                )}

                {/* Service Image */}
                <div className="relative h-48 overflow-hidden">
                  <img
                    src={service.image || "/placeholder.svg"}
                    alt={service.name}
                    className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-500"
                  />
                  <div className="absolute inset-0 bg-gradient-to-t from-black/50 to-transparent"></div>
                  <div className="absolute bottom-4 left-4">
                    <div className="bg-white/20 backdrop-blur-sm rounded-full p-3">
                      <IconComponent className="w-6 h-6 text-white" />
                    </div>
                  </div>
                </div>

                {/* Content */}
                <div className="p-6">
                  <div className="mb-4">
                    <Badge variant="outline" className="text-xs mb-2 border-purple-200 text-purple-600">
                      {service.category}
                    </Badge>
                    <h3 className="text-xl font-bold text-gray-900 mb-2 group-hover:text-purple-600 transition-colors">
                      {service.name}
                    </h3>
                    <p className="text-gray-600 text-sm leading-relaxed">{service.description}</p>
                  </div>

                  {/* Features */}
                  <div className="mb-6">
                    <h4 className="text-sm font-semibold text-gray-900 mb-2">Key Features:</h4>
                    <ul className="space-y-1">
                      {service.features.slice(0, 3).map((feature, index) => (
                        <li key={index} className="flex items-center text-xs text-gray-600">
                          <Check className="w-3 h-3 text-green-500 mr-2 flex-shrink-0" />
                          {feature}
                        </li>
                      ))}
                    </ul>
                  </div>

                  {/* Pricing */}
                  <div className="mb-6">
                    <div className="flex items-center gap-2 mb-2">
                      <span className="text-2xl font-bold text-gray-900">{formatPrice(service.price)}</span>
                      <span className="text-sm text-gray-500 line-through">{formatPrice(service.originalPrice)}</span>
                    </div>
                    <div className="text-xs text-green-600 font-medium">
                      Save {formatPrice(service.originalPrice - service.price)}
                    </div>
                  </div>

                  {/* Add to Cart Button */}
                  <Button
                    onClick={() => addToCart(service)}
                    disabled={isInCart(service.id)}
                    className={`w-full py-3 rounded-xl font-semibold transition-all duration-300 ${
                      isInCart(service.id)
                        ? "bg-green-500 hover:bg-green-600 text-white"
                        : "bg-gradient-to-r from-purple-600 to-blue-600 hover:from-purple-700 hover:to-blue-700 text-white shadow-lg hover:shadow-xl transform hover:scale-105"
                    }`}
                  >
                    {isInCart(service.id) ? (
                      <>
                        <Check className="w-4 h-4 mr-2" />
                        Added to Cart
                      </>
                    ) : (
                      <>
                        <Plus className="w-4 h-4 mr-2" />
                        Add to Cart
                      </>
                    )}
                  </Button>
                </div>
              </div>
            )
          })}
        </div>
      </div>
    </section>
  )
}
