// Database operations for content management
// This example uses a simple JSON file approach, but you can replace with any database

import fs from 'fs/promises'
import path from 'path'
import crypto from 'crypto'

const DATA_DIR = path.join(process.cwd(), 'data')

// Ensure data directory exists
export async function ensureDataDir() {
  try {
    await fs.access(DATA_DIR)
  } catch {
    await fs.mkdir(DATA_DIR, { recursive: true })
  }
}

// Generic CRUD operations
export async function readData<T>(filename: string): Promise<T[]> {
  await ensureDataDir()
  const filePath = path.join(DATA_DIR, `${filename}.json`)
  
  try {
    const data = await fs.readFile(filePath, 'utf-8')
    return JSON.parse(data)
  } catch {
    return []
  }
}

export async function writeData<T>(filename: string, data: T[]): Promise<void> {
  await ensureDataDir()
  const filePath = path.join(DATA_DIR, `${filename}.json`)
  await fs.writeFile(filePath, JSON.stringify(data, null, 2))
}

export async function createRecord<T extends { id?: string }>(filename: string, record: Omit<T, 'id' | 'createdAt' | 'updatedAt'>): Promise<T> {
  const data = await readData<T>(filename)
  const newRecord = {
    ...record,
    id: (crypto as any).randomUUID ? crypto.randomUUID() : Math.random().toString(36).slice(2),
    createdAt: new Date(),
    updatedAt: new Date()
  } as T
  
  data.push(newRecord)
  await writeData(filename, data)
  return newRecord
}

export async function updateRecord<T extends { id: string }>(filename: string, id: string, updates: Partial<T>): Promise<T | null> {
  const data = await readData<T>(filename)
  const index = data.findIndex(item => item.id === id)
  
  if (index === -1) return null
  
  data[index] = {
    ...data[index],
    ...updates,
    updatedAt: new Date()
  } as T
  
  await writeData(filename, data)
  return data[index]
}

export async function deleteRecord<T extends { id: string }>(filename: string, id: string): Promise<boolean> {
  const data = await readData<T>(filename)
  const index = data.findIndex(item => item.id === id)
  
  if (index === -1) return false
  
  data.splice(index, 1)
  await writeData(filename, data)
  return true
}

export async function findRecord<T extends { id: string }>(filename: string, id: string): Promise<T | null> {
  const data = await readData<T>(filename)
  return data.find(item => item.id === id) || null
}

// Specific content operations
export const ContentDB = {
  hero: {
    get: () => readData('hero-content'),
    create: (data: any) => createRecord('hero-content', data),
    update: (id: string, data: any) => updateRecord('hero-content', id, data),
    delete: (id: string) => deleteRecord('hero-content', id)
  },
  about: {
    get: () => readData('about-content'),
    create: (data: any) => createRecord('about-content', data),
    update: (id: string, data: any) => updateRecord('about-content', id, data),
    delete: (id: string) => deleteRecord('about-content', id)
  },
  services: {
    get: () => readData('services'),
    create: (data: any) => createRecord('services', data),
    update: (id: string, data: any) => updateRecord('services', id, data),
    delete: (id: string) => deleteRecord('services', id)
  },
  team: {
    get: () => readData('team-members'),
    create: (data: any) => createRecord('team-members', data),
    update: (id: string, data: any) => updateRecord('team-members', id, data),
    delete: (id: string) => deleteRecord('team-members', id)
  },
  contact: {
    get: () => readData('contact-info'),
    create: (data: any) => createRecord('contact-info', data),
    update: (id: string, data: any) => updateRecord('contact-info', id, data),
    delete: (id: string) => deleteRecord('contact-info', id)
  },
  settings: {
    get: () => readData('site-settings'),
    create: (data: any) => createRecord('site-settings', data),
    update: (id: string, data: any) => updateRecord('site-settings', id, data),
    delete: (id: string) => deleteRecord('site-settings', id)
  },
  users: {
    get: () => readData('users'),
    create: (data: any) => createRecord('users', data),
    update: (id: string, data: any) => updateRecord('users', id, data),
    delete: (id: string) => deleteRecord('users', id)
  },
  quotes: {
    get: () => readData('quote-requests'),
    create: (data: any) => createRecord('quote-requests', data),
    update: (id: string, data: any) => updateRecord('quote-requests', id, data),
    delete: (id: string) => deleteRecord('quote-requests', id)
  },
  media: {
    get: () => readData('media-assets'),
    create: (data: any) => createRecord('media-assets', data),
    update: (id: string, data: any) => updateRecord('media-assets', id, data),
    delete: (id: string) => deleteRecord('media-assets', id)
  }
}
