"use client"

import type React from "react"

import { useState, useEffect } from "react"
import { useRouter } from "next/navigation"
import { AdminLayout } from "@/components/admin/admin-layout"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Textarea } from "@/components/ui/textarea"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Label } from "@/components/ui/label"
import { Save, RefreshCw, Upload, Download, Palette, ImageIcon, Navigation, Settings, FileText } from "lucide-react"
import { useToast } from "@/components/ui/use-toast"

interface ContentData {
  hero?: {
    title: string
    subtitle: string
    ctaText: string
    backgroundImage?: string
  }
  about?: {
    title: string
    description: string
    mission: string
    vision: string
  }
  siteInfo?: {
    companyName: string
    tagline: string
    phone: string
    email: string
    address: string
  }
  services?: {
    title: string
    subtitle: string
  }
}

export default function ContentManagement() {
  const router = useRouter()
  const { toast } = useToast()
  const [loading, setLoading] = useState(true)
  const [saving, setSaving] = useState(false)
  const [contentData, setContentData] = useState<ContentData>({})

  useEffect(() => {
    const isLoggedIn = localStorage.getItem("adminLoggedIn")
    if (isLoggedIn !== "true") {
      router.push("/admin")
      return
    }
    loadContent()
  }, [router])

  const loadContent = async () => {
    try {
      setLoading(true)
      const token = localStorage.getItem("adminToken") || "admin-token-placeholder"
      
      // Load hero content
      const heroResponse = await fetch('/api/admin/content?type=hero', {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      })
      
      // Load about content
      const aboutResponse = await fetch('/api/admin/content?type=about', {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      })

      const heroData = heroResponse.ok ? await heroResponse.json() : null
      const aboutData = aboutResponse.ok ? await aboutResponse.json() : null

      setContentData({
        hero: heroData || {
          title: "Comprehensive Insurance Solutions for Modern Nigeria",
          subtitle: "Protecting what matters most with innovative insurance products tailored for individuals and businesses across Nigeria.",
          ctaText: "Get Your Quote Today"
        },
        about: aboutData || {
          title: "About Bishop & Knight",
          description: "We are a dynamic and fully licensed insurance brokerage firm...",
          mission: "To provide exceptional insurance solutions...",
          vision: "To be Nigeria's leading insurance brokerage firm..."
        },
        siteInfo: {
          companyName: "Bishop & Knight Insurance Brokers Limited",
          tagline: "Your Trusted Insurance Partner",
          phone: "+234-XXX-XXX-XXXX",
          email: "info@bishopknight.com",
          address: "Lagos, Nigeria"
        },
        services: {
          title: "Our Insurance Services",
          subtitle: "Comprehensive coverage for all your needs"
        }
      })
    } catch (error) {
      console.error('Error loading content:', error)
      toast({
        title: "Error",
        description: "Failed to load content. Using default values.",
        variant: "destructive"
      })
    } finally {
      setLoading(false)
    }
  }

  const handleSave = async () => {
    try {
      setSaving(true)
      const token = localStorage.getItem("adminToken") || "admin-token-placeholder"
      
      // Save hero content
      if (contentData.hero) {
        const heroResponse = await fetch('/api/admin/content', {
          method: 'POST',
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
          },
          body: JSON.stringify({
            type: 'hero',
            data: contentData.hero
          })
        })
        
        if (!heroResponse.ok) {
          throw new Error('Failed to save hero content')
        }
      }

      // Save about content
      if (contentData.about) {
        const aboutResponse = await fetch('/api/admin/content', {
          method: 'POST',
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
          },
          body: JSON.stringify({
            type: 'about',
            data: contentData.about
          })
        })
        
        if (!aboutResponse.ok) {
          throw new Error('Failed to save about content')
        }
      }

      toast({
        title: "Success",
        description: "Content saved successfully!",
      })
    } catch (error) {
      console.error('Error saving content:', error)
      toast({
        title: "Error",
        description: "Failed to save content. Please try again.",
        variant: "destructive"
      })
    } finally {
      setSaving(false)
    }
  }

  const handleExport = () => {
    const dataStr = JSON.stringify(contentData, null, 2)
    const dataBlob = new Blob([dataStr], { type: "application/json" })
    const url = URL.createObjectURL(dataBlob)
    const link = document.createElement("a")
    link.href = url
    link.download = "site-content-backup.json"
    link.click()
  }

  const handleImport = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0]
    if (file) {
      const reader = new FileReader()
      reader.onload = (e) => {
        try {
          const importedData = JSON.parse(e.target?.result as string)
          setContentData(importedData)
          toast({
            title: "Success",
            description: "Content imported successfully!",
          })
        } catch (error) {
          toast({
            title: "Error",
            description: "Invalid JSON file",
            variant: "destructive"
          })
        }
      }
      reader.readAsText(file)
    }
  }

  if (loading) {
    return (
      <AdminLayout>
        <div className="flex items-center justify-center h-64">
          <div className="text-white text-xl flex items-center">
            <RefreshCw className="w-6 h-6 mr-2 animate-spin" />
            Loading content...
          </div>
        </div>
      </AdminLayout>
    )
  }

  return (
    <AdminLayout>
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-white">Content Management</h1>
            <p className="text-gray-300">Edit website content with real-time API integration</p>
          </div>

          <div className="flex items-center space-x-2">
            <Button
              variant="outline"
              onClick={handleExport}
              className="border-blue-500/30 text-blue-400 hover:bg-blue-500/10 bg-transparent"
            >
              <Download className="w-4 h-4 mr-2" />
              Export
            </Button>

            <label className="cursor-pointer">
              <Button
                variant="outline"
                className="border-green-500/30 text-green-400 hover:bg-green-500/10 bg-transparent"
                asChild
              >
                <span>
                  <Upload className="w-4 h-4 mr-2" />
                  Import
                </span>
              </Button>
              <input type="file" accept=".json" onChange={handleImport} className="hidden" />
            </label>

            <Button
              onClick={handleSave}
              disabled={saving}
              className="bg-gradient-to-r from-purple-500 to-blue-500 hover:from-purple-600 hover:to-blue-600"
            >
              {saving ? (
                <>
                  <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                  Saving...
                </>
              ) : (
                <>
                  <Save className="w-4 h-4 mr-2" />
                  Save Changes
                </>
              )}
            </Button>
          </div>
        </div>

        <Tabs defaultValue="pages" className="space-y-6">
          <TabsList className="bg-white/10 border-white/20">
            <TabsTrigger value="pages" className="data-[state=active]:bg-purple-500/20">
              <FileText className="w-4 h-4 mr-2" />
              Pages
            </TabsTrigger>
            <TabsTrigger value="general" className="data-[state=active]:bg-purple-500/20">
              <Settings className="w-4 h-4 mr-2" />
              General
            </TabsTrigger>
          </TabsList>

          <TabsContent value="pages" className="space-y-6">
            {/* Hero Section */}
            <Card className="bg-white/10 backdrop-blur-lg border-white/20">
              <CardHeader>
                <CardTitle className="text-white">Hero Section</CardTitle>
                <CardDescription className="text-gray-300">Main homepage hero content</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-2">
                  <Label className="text-white">Hero Title</Label>
                  <Input
                    value={contentData.hero?.title || ""}
                    onChange={(e) =>
                      setContentData({
                        ...contentData,
                        hero: { ...contentData.hero!, title: e.target.value },
                      })
                    }
                    className="bg-white/10 border-white/20 text-white"
                    placeholder="Enter hero title"
                  />
                </div>

                <div className="space-y-2">
                  <Label className="text-white">Hero Subtitle</Label>
                  <Textarea
                    value={contentData.hero?.subtitle || ""}
                    onChange={(e) =>
                      setContentData({
                        ...contentData,
                        hero: { ...contentData.hero!, subtitle: e.target.value },
                      })
                    }
                    className="bg-white/10 border-white/20 text-white"
                    rows={3}
                    placeholder="Enter hero subtitle"
                  />
                </div>

                <div className="space-y-2">
                  <Label className="text-white">CTA Button Text</Label>
                  <Input
                    value={contentData.hero?.ctaText || ""}
                    onChange={(e) =>
                      setContentData({
                        ...contentData,
                        hero: { ...contentData.hero!, ctaText: e.target.value },
                      })
                    }
                    className="bg-white/10 border-white/20 text-white"
                    placeholder="Enter CTA button text"
                  />
                </div>
              </CardContent>
            </Card>

            {/* About Section */}
            <Card className="bg-white/10 backdrop-blur-lg border-white/20">
              <CardHeader>
                <CardTitle className="text-white">About Section</CardTitle>
                <CardDescription className="text-gray-300">About us content and company information</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-2">
                  <Label className="text-white">About Title</Label>
                  <Input
                    value={contentData.about?.title || ""}
                    onChange={(e) =>
                      setContentData({
                        ...contentData,
                        about: { ...contentData.about!, title: e.target.value },
                      })
                    }
                    className="bg-white/10 border-white/20 text-white"
                    placeholder="Enter about title"
                  />
                </div>

                <div className="space-y-2">
                  <Label className="text-white">Description</Label>
                  <Textarea
                    value={contentData.about?.description || ""}
                    onChange={(e) =>
                      setContentData({
                        ...contentData,
                        about: { ...contentData.about!, description: e.target.value },
                      })
                    }
                    className="bg-white/10 border-white/20 text-white"
                    rows={4}
                    placeholder="Enter about description"
                  />
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label className="text-white">Mission</Label>
                    <Textarea
                      value={contentData.about?.mission || ""}
                      onChange={(e) =>
                        setContentData({
                          ...contentData,
                          about: { ...contentData.about!, mission: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                      rows={3}
                      placeholder="Enter mission statement"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">Vision</Label>
                    <Textarea
                      value={contentData.about?.vision || ""}
                      onChange={(e) =>
                        setContentData({
                          ...contentData,
                          about: { ...contentData.about!, vision: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                      rows={3}
                      placeholder="Enter vision statement"
                    />
                  </div>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="general" className="space-y-6">
            <Card className="bg-white/10 backdrop-blur-lg border-white/20">
              <CardHeader>
                <CardTitle className="text-white">Site Information</CardTitle>
                <CardDescription className="text-gray-300">Basic company and contact information</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label className="text-white">Company Name</Label>
                    <Input
                      value={contentData.siteInfo?.companyName || ""}
                      onChange={(e) =>
                        setContentData({
                          ...contentData,
                          siteInfo: { ...contentData.siteInfo!, companyName: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">Tagline</Label>
                    <Input
                      value={contentData.siteInfo?.tagline || ""}
                      onChange={(e) =>
                        setContentData({
                          ...contentData,
                          siteInfo: { ...contentData.siteInfo!, tagline: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">Phone</Label>
                    <Input
                      value={contentData.siteInfo?.phone || ""}
                      onChange={(e) =>
                        setContentData({
                          ...contentData,
                          siteInfo: { ...contentData.siteInfo!, phone: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label className="text-white">Email</Label>
                    <Input
                      value={contentData.siteInfo?.email || ""}
                      onChange={(e) =>
                        setContentData({
                          ...contentData,
                          siteInfo: { ...contentData.siteInfo!, email: e.target.value },
                        })
                      }
                      className="bg-white/10 border-white/20 text-white"
                    />
                  </div>
                </div>

                <div className="space-y-2">
                  <Label className="text-white">Address</Label>
                  <Textarea
                    value={contentData.siteInfo?.address || ""}
                    onChange={(e) =>
                      setContentData({
                        ...contentData,
                        siteInfo: { ...contentData.siteInfo!, address: e.target.value },
                      })
                    }
                    className="bg-white/10 border-white/20 text-white"
                    rows={3}
                  />
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </AdminLayout>
  )
}
