"use client"

import type React from "react"

import { useState, useEffect, useCallback } from "react"
import { useRouter } from "next/navigation"
import { AdminLayout } from "@/components/admin/admin-layout"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog"
import { Label } from "@/components/ui/label"
import {
  ImageIcon,
  Upload,
  Search,
  Trash2,
  FolderPlus,
  Grid,
  List,
  MoreHorizontal,
  FileText,
  Video,
  Music,
  Archive,
  RefreshCw,
} from "lucide-react"

interface MediaFile {
  id: string
  name: string
  type: "image" | "video" | "audio" | "document" | "other"
  size: number
  url: string
  thumbnail?: string
  uploadDate: string
  folder: string
  dimensions?: { width: number; height: number }
  usedIn: string[]
}

interface MediaFolder {
  id: string
  name: string
  fileCount: number
  createdDate: string
}

export default function MediaManagement() {
  const router = useRouter()
  const [loading, setLoading] = useState(true)
  const [viewMode, setViewMode] = useState<"grid" | "list">("grid")
  const [searchTerm, setSearchTerm] = useState("")
  const [selectedFolder, setSelectedFolder] = useState("all")
  const [selectedFiles, setSelectedFiles] = useState<string[]>([])
  const [isDragOver, setIsDragOver] = useState(false)
  const [isUploading, setIsUploading] = useState(false)

  const [folders, setFolders] = useState<MediaFolder[]>([
    { id: "logos", name: "Logos & Branding", fileCount: 3, createdDate: "2024-01-15" },
    { id: "hero", name: "Hero Media", fileCount: 2, createdDate: "2024-01-15" },
    { id: "team", name: "Team Photos", fileCount: 8, createdDate: "2024-01-16" },
    { id: "services", name: "Service Images", fileCount: 12, createdDate: "2024-01-17" },
    { id: "documents", name: "Documents", fileCount: 5, createdDate: "2024-01-18" },
  ])

  const [mediaFiles, setMediaFiles] = useState<MediaFile[]>([
    {
      id: "1",
      name: "company-logo.png",
      type: "image",
      size: 45000,
      url: "/logo.png",
      thumbnail: "/logo.png",
      uploadDate: "2024-01-15",
      folder: "logos",
      dimensions: { width: 200, height: 80 },
      usedIn: ["Header", "Footer", "About Page"],
    },
    {
      id: "2",
      name: "hero-video.mp4",
      type: "video",
      size: 15000000,
      url: "/hero.mp4",
      thumbnail: "/hero-thumb.jpg",
      uploadDate: "2024-01-15",
      folder: "hero",
      usedIn: ["Homepage Hero"],
    },
    {
      id: "3",
      name: "favicon.ico",
      type: "image",
      size: 2048,
      url: "/favicon.ico",
      uploadDate: "2024-01-15",
      folder: "logos",
      dimensions: { width: 32, height: 32 },
      usedIn: ["Site Icon"],
    },
    {
      id: "4",
      name: "team-photo-1.jpg",
      type: "image",
      size: 120000,
      url: "/team/photo1.jpg",
      thumbnail: "/team/photo1.jpg",
      uploadDate: "2024-01-16",
      folder: "team",
      dimensions: { width: 400, height: 400 },
      usedIn: ["Leadership Page"],
    },
  ])

  useEffect(() => {
    const isLoggedIn = localStorage.getItem("adminLoggedIn")
    if (isLoggedIn !== "true") {
      router.push("/admin")
      return
    }
    setLoading(false)
  }, [router])

  const filteredFiles = mediaFiles.filter((file) => {
    const matchesSearch = file.name.toLowerCase().includes(searchTerm.toLowerCase())
    const matchesFolder = selectedFolder === "all" || file.folder === selectedFolder
    return matchesSearch && matchesFolder
  })

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return "0 Bytes"
    const k = 1024
    const sizes = ["Bytes", "KB", "MB", "GB"]
    const i = Math.floor(Math.log(bytes) / Math.log(k))
    return Number.parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + " " + sizes[i]
  }

  const getFileIcon = (type: string) => {
    switch (type) {
      case "image":
        return <ImageIcon className="w-5 h-5" />
      case "video":
        return <Video className="w-5 h-5" />
      case "audio":
        return <Music className="w-5 h-5" />
      case "document":
        return <FileText className="w-5 h-5" />
      default:
        return <Archive className="w-5 h-5" />
    }
  }

  const handleDragOver = useCallback((e: React.DragEvent) => {
    e.preventDefault()
    setIsDragOver(true)
  }, [])

  const handleDragLeave = useCallback((e: React.DragEvent) => {
    e.preventDefault()
    setIsDragOver(false)
  }, [])

  const handleDrop = useCallback((e: React.DragEvent) => {
    e.preventDefault()
    setIsDragOver(false)
    const files = Array.from(e.dataTransfer.files)
    handleFileUpload(files)
  }, [])

  const handleFileUpload = async (files: File[]) => {
    setIsUploading(true)

    // Simulate file upload
    for (const file of files) {
      const newFile: MediaFile = {
        id: Date.now().toString() + Math.random().toString(36).substr(2, 9),
        name: file.name,
        type: file.type.startsWith("image/")
          ? "image"
          : file.type.startsWith("video/")
            ? "video"
            : file.type.startsWith("audio/")
              ? "audio"
              : file.type.includes("pdf") || file.type.includes("document")
                ? "document"
                : "other",
        size: file.size,
        url: URL.createObjectURL(file),
        thumbnail: file.type.startsWith("image/") ? URL.createObjectURL(file) : undefined,
        uploadDate: new Date().toISOString().split("T")[0],
        folder: selectedFolder === "all" ? "general" : selectedFolder,
        usedIn: [],
      }

      setMediaFiles((prev) => [...prev, newFile])
    }

    setIsUploading(false)
  }

  const handleDeleteFiles = () => {
    if (confirm(`Are you sure you want to delete ${selectedFiles.length} file(s)?`)) {
      setMediaFiles((prev) => prev.filter((file) => !selectedFiles.includes(file.id)))
      setSelectedFiles([])
    }
  }

  const handleSelectFile = (fileId: string) => {
    setSelectedFiles((prev) => (prev.includes(fileId) ? prev.filter((id) => id !== fileId) : [...prev, fileId]))
  }

  if (loading) {
    return (
      <AdminLayout>
        <div className="flex items-center justify-center h-64">
          <div className="text-white text-xl">Loading media...</div>
        </div>
      </AdminLayout>
    )
  }

  return (
    <AdminLayout>
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-white">Media Management</h1>
            <p className="text-gray-300">Upload, organize, and manage all media files</p>
          </div>

          <div className="flex items-center space-x-2">
            <Button
              variant="outline"
              onClick={() => setViewMode(viewMode === "grid" ? "list" : "grid")}
              className="border-gray-500/30 text-gray-400 hover:bg-gray-500/10 bg-transparent"
            >
              {viewMode === "grid" ? <List className="w-4 h-4" /> : <Grid className="w-4 h-4" />}
            </Button>

            <Dialog>
              <DialogTrigger asChild>
                <Button
                  variant="outline"
                  className="border-green-500/30 text-green-400 hover:bg-green-500/10 bg-transparent"
                >
                  <FolderPlus className="w-4 h-4 mr-2" />
                  New Folder
                </Button>
              </DialogTrigger>
              <DialogContent className="bg-slate-900 border-white/20">
                <DialogHeader>
                  <DialogTitle className="text-white">Create New Folder</DialogTitle>
                  <DialogDescription className="text-gray-300">
                    Create a new folder to organize your media files
                  </DialogDescription>
                </DialogHeader>
                <div className="space-y-4">
                  <div className="space-y-2">
                    <Label className="text-white">Folder Name</Label>
                    <Input className="bg-white/10 border-white/20 text-white" placeholder="Enter folder name" />
                  </div>
                  <Button className="w-full bg-gradient-to-r from-purple-500 to-blue-500">Create Folder</Button>
                </div>
              </DialogContent>
            </Dialog>

            <label className="cursor-pointer">
              <Button className="bg-gradient-to-r from-purple-500 to-blue-500 hover:from-purple-600 hover:to-blue-600">
                <Upload className="w-4 h-4 mr-2" />
                Upload Files
              </Button>
              <input
                type="file"
                multiple
                accept="image/*,video/*,audio/*,.pdf,.doc,.docx"
                onChange={(e) => e.target.files && handleFileUpload(Array.from(e.target.files))}
                className="hidden"
              />
            </label>
          </div>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
          <Card className="bg-white/10 backdrop-blur-lg border-white/20">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-300 text-sm">Total Files</p>
                  <p className="text-2xl font-bold text-white">{mediaFiles.length}</p>
                </div>
                <ImageIcon className="w-8 h-8 text-purple-400" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-white/10 backdrop-blur-lg border-white/20">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-300 text-sm">Images</p>
                  <p className="text-2xl font-bold text-white">{mediaFiles.filter((f) => f.type === "image").length}</p>
                </div>
                <ImageIcon className="w-8 h-8 text-blue-400" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-white/10 backdrop-blur-lg border-white/20">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-300 text-sm">Videos</p>
                  <p className="text-2xl font-bold text-white">{mediaFiles.filter((f) => f.type === "video").length}</p>
                </div>
                <Video className="w-8 h-8 text-green-400" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-white/10 backdrop-blur-lg border-white/20">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-300 text-sm">Storage Used</p>
                  <p className="text-2xl font-bold text-white">
                    {formatFileSize(mediaFiles.reduce((sum, f) => sum + f.size, 0))}
                  </p>
                </div>
                <Archive className="w-8 h-8 text-orange-400" />
              </div>
            </CardContent>
          </Card>
        </div>

        <div className="flex flex-col lg:flex-row gap-6">
          {/* Sidebar */}
          <div className="lg:w-64 space-y-4">
            <Card className="bg-white/10 backdrop-blur-lg border-white/20">
              <CardHeader>
                <CardTitle className="text-white text-lg">Folders</CardTitle>
              </CardHeader>
              <CardContent className="space-y-2">
                <Button
                  variant={selectedFolder === "all" ? "default" : "ghost"}
                  className="w-full justify-start text-white"
                  onClick={() => setSelectedFolder("all")}
                >
                  All Files ({mediaFiles.length})
                </Button>
                {folders.map((folder) => (
                  <Button
                    key={folder.id}
                    variant={selectedFolder === folder.id ? "default" : "ghost"}
                    className="w-full justify-start text-white"
                    onClick={() => setSelectedFolder(folder.id)}
                  >
                    {folder.name} ({folder.fileCount})
                  </Button>
                ))}
              </CardContent>
            </Card>
          </div>

          {/* Main Content */}
          <div className="flex-1 space-y-4">
            <Card className="bg-white/10 backdrop-blur-lg border-white/20">
              <CardContent className="p-4">
                <div className="flex flex-col md:flex-row gap-4">
                  <div className="flex-1">
                    <div className="relative">
                      <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                      <Input
                        placeholder="Search files..."
                        value={searchTerm}
                        onChange={(e) => setSearchTerm(e.target.value)}
                        className="pl-10 bg-white/10 border-white/20 text-white"
                      />
                    </div>
                  </div>

                  {selectedFiles.length > 0 && (
                    <div className="flex items-center space-x-2">
                      <span className="text-white text-sm">{selectedFiles.length} selected</span>
                      <Button
                        size="sm"
                        variant="outline"
                        className="border-red-500/30 text-red-400 hover:bg-red-500/10 bg-transparent"
                        onClick={handleDeleteFiles}
                      >
                        <Trash2 className="w-4 h-4" />
                      </Button>
                    </div>
                  )}
                </div>
              </CardContent>
            </Card>

            {/* Upload Area */}
            <div
              className={`border-2 border-dashed rounded-lg p-8 text-center transition-colors ${
                isDragOver ? "border-purple-500 bg-purple-500/10" : "border-white/20 bg-white/5"
              }`}
              onDragOver={handleDragOver}
              onDragLeave={handleDragLeave}
              onDrop={handleDrop}
            >
              {isUploading ? (
                <div className="flex items-center justify-center space-x-2">
                  <RefreshCw className="w-6 h-6 text-purple-400 animate-spin" />
                  <span className="text-white">Uploading files...</span>
                </div>
              ) : (
                <div>
                  <Upload className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                  <p className="text-white mb-2">Drag and drop files here, or click to browse</p>
                  <p className="text-gray-400 text-sm">Supports images, videos, audio, and documents</p>
                </div>
              )}
            </div>

            {/* File Grid/List */}
            <Card className="bg-white/10 backdrop-blur-lg border-white/20">
              <CardContent className="p-6">
                {viewMode === "grid" ? (
                  <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 xl:grid-cols-6 gap-4">
                    {filteredFiles.map((file) => (
                      <div
                        key={file.id}
                        className={`relative group cursor-pointer rounded-lg border-2 transition-colors ${
                          selectedFiles.includes(file.id)
                            ? "border-purple-500 bg-purple-500/10"
                            : "border-white/20 bg-white/5 hover:border-white/40"
                        }`}
                        onClick={() => handleSelectFile(file.id)}
                      >
                        <div className="aspect-square p-2">
                          {file.type === "image" && file.thumbnail ? (
                            <img
                              src={file.thumbnail || "/placeholder.svg"}
                              alt={file.name}
                              className="w-full h-full object-cover rounded"
                            />
                          ) : (
                            <div className="w-full h-full flex items-center justify-center bg-white/10 rounded">
                              {getFileIcon(file.type)}
                            </div>
                          )}
                        </div>
                        <div className="p-2">
                          <p className="text-white text-sm font-medium truncate">{file.name}</p>
                          <p className="text-gray-400 text-xs">{formatFileSize(file.size)}</p>
                        </div>
                        <div className="absolute top-2 right-2 opacity-0 group-hover:opacity-100 transition-opacity">
                          <Button size="sm" variant="ghost" className="h-6 w-6 p-0 bg-black/50 hover:bg-black/70">
                            <MoreHorizontal className="w-3 h-3 text-white" />
                          </Button>
                        </div>
                      </div>
                    ))}
                  </div>
                ) : (
                  <div className="space-y-2">
                    {filteredFiles.map((file) => (
                      <div
                        key={file.id}
                        className={`flex items-center space-x-4 p-3 rounded-lg border transition-colors cursor-pointer ${
                          selectedFiles.includes(file.id)
                            ? "border-purple-500 bg-purple-500/10"
                            : "border-white/20 bg-white/5 hover:border-white/40"
                        }`}
                        onClick={() => handleSelectFile(file.id)}
                      >
                        <div className="flex-shrink-0">
                          {file.type === "image" && file.thumbnail ? (
                            <img
                              src={file.thumbnail || "/placeholder.svg"}
                              alt={file.name}
                              className="w-10 h-10 object-cover rounded"
                            />
                          ) : (
                            <div className="w-10 h-10 flex items-center justify-center bg-white/10 rounded">
                              {getFileIcon(file.type)}
                            </div>
                          )}
                        </div>
                        <div className="flex-1 min-w-0">
                          <p className="text-white font-medium truncate">{file.name}</p>
                          <p className="text-gray-400 text-sm">
                            {formatFileSize(file.size)} • {file.uploadDate}
                          </p>
                        </div>
                        <div className="flex-shrink-0">
                          <Badge variant="secondary">{file.type}</Badge>
                        </div>
                        <div className="flex-shrink-0">
                          <Button size="sm" variant="ghost" className="text-gray-400 hover:text-white">
                            <MoreHorizontal className="w-4 h-4" />
                          </Button>
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </AdminLayout>
  )
}
