"use client"

import type React from "react"

import { useState, useEffect } from "react"
import { useRouter } from "next/navigation"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Label } from "@/components/ui/label"
import { Alert, AlertDescription } from "@/components/ui/alert"
import { Shield, Lock, Eye, EyeOff } from "lucide-react"

export default function AdminLogin() {
  const [username, setUsername] = useState("")
  const [password, setPassword] = useState("")
  const [showPassword, setShowPassword] = useState(false)
  const [error, setError] = useState("")
  const [loading, setLoading] = useState(false)
  const [mounted, setMounted] = useState(false) // Add this
  const router = useRouter()

  useEffect(() => {
    setMounted(true) // Add this
    const isLoggedIn = localStorage.getItem("adminLoggedIn")
    if (isLoggedIn === "true") {
      router.push("/admin/dashboard")
    }
  }, [router])

  const handleLogin = async (e: React.FormEvent) => {
    e.preventDefault()
    setLoading(true)
    setError("")

    try {
      const storedUsername = localStorage.getItem("adminUsername") || "Admin"
      const storedPassword = localStorage.getItem("adminPassword") || "Admin"
      
      // Add debug logging
      console.log('Login attempt:', {
        inputUsername: username,
        inputPassword: password,
        storedUsername,
        storedPassword,
        usernameMatch: username === storedUsername,
        passwordMatch: password === storedPassword
      })
  
      // Simulate authentication delay
      await new Promise((resolve) => setTimeout(resolve, 1000))
  
      if (username === storedUsername && password === storedPassword) {
        // Set localStorage values
        localStorage.setItem("adminLoggedIn", "true")
        localStorage.setItem("adminLoginTime", new Date().toISOString())
        
        // FIXED: Also set cookie for middleware
        document.cookie = "adminLoggedIn=true; path=/; max-age=86400" // 24 hours
        
        console.log('Login successful, redirecting...')
        
        // Force a small delay to ensure localStorage and cookie are set
        await new Promise((resolve) => setTimeout(resolve, 200))
        
        // Navigate to dashboard
        router.push("/admin/dashboard")
      } else {
        console.log('Login failed: credentials mismatch')
        setError("Invalid username or password. Try: Admin/Admin (case-sensitive)")
      }
    } catch (error) {
      console.error("Login error:", error)
      setError("An error occurred during login")
    } finally {
      setLoading(false)
    }
  }

  // Generate consistent positions based on index
  const generatePosition = (index: number) => {
    const seed = index * 12345 // Use index as seed for consistent positioning
    return {
      left: `${(seed % 100)}%`,
      top: `${((seed * 7) % 100)}%`,
      animationDelay: `${(seed % 3)}s`,
      animationDuration: `${2 + (seed % 3)}s`,
    }
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-900 via-purple-900 to-slate-900 flex items-center justify-center p-4">
      <div className="absolute inset-0 overflow-hidden">
        <div className="absolute -inset-10 opacity-50">
          {mounted && [...Array(50)].map((_, i) => (
            <div
              key={i}
              className="absolute animate-pulse"
              style={generatePosition(i)}
            >
              <div className="w-1 h-1 bg-purple-400 rounded-full"></div>
            </div>
          ))}
        </div>
      </div>

      <Card className="w-full max-w-md relative z-10 bg-white/10 backdrop-blur-lg border-white/20">
        <CardHeader className="text-center space-y-4">
          <div className="mx-auto w-16 h-16 bg-gradient-to-r from-purple-500 to-blue-500 rounded-full flex items-center justify-center">
            <Shield className="w-8 h-8 text-white" />
          </div>
          <div>
            <CardTitle className="text-2xl font-bold text-white">Admin Portal</CardTitle>
            <CardDescription className="text-gray-300">Bishop & Knight Insurance Brokers</CardDescription>
          </div>
        </CardHeader>

        <CardContent>
          <form onSubmit={handleLogin} className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="username" className="text-white">
                Username
              </Label>
              <Input
                id="username"
                type="text"
                value={username}
                onChange={(e) => setUsername(e.target.value)}
                className="bg-white/10 border-white/20 text-white placeholder:text-gray-400"
                placeholder="Enter username"
                required
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="password" className="text-white">
                Password
              </Label>
              <div className="relative">
                <Input
                  id="password"
                  type={showPassword ? "text" : "password"}
                  value={password}
                  onChange={(e) => setPassword(e.target.value)}
                  className="bg-white/10 border-white/20 text-white placeholder:text-gray-400 pr-10"
                  placeholder="Enter password"
                  required
                />
                <Button
                  type="button"
                  variant="ghost"
                  size="sm"
                  className="absolute right-0 top-0 h-full px-3 text-gray-400 hover:text-white"
                  onClick={() => setShowPassword(!showPassword)}
                >
                  {showPassword ? <EyeOff className="w-4 h-4" /> : <Eye className="w-4 h-4" />}
                </Button>
              </div>
            </div>

            {error && (
              <Alert className="bg-red-500/10 border-red-500/20">
                <AlertDescription className="text-red-400">{error}</AlertDescription>
              </Alert>
            )}

            <Button
              type="submit"
              className="w-full bg-gradient-to-r from-purple-500 to-blue-500 hover:from-purple-600 hover:to-blue-600"
              disabled={loading}
            >
              {loading ? (
                <div className="flex items-center space-x-2">
                  <div className="w-4 h-4 border-2 border-white/30 border-t-white rounded-full animate-spin"></div>
                  <span>Authenticating...</span>
                </div>
              ) : (
                <div className="flex items-center space-x-2">
                  <Lock className="w-4 h-4" />
                  <span>Sign In</span>
                </div>
              )}
            </Button>
          </form>

          <div className="mt-6 text-center">
            <p className="text-xs text-gray-400">Default: Username: Admin, Password: Admin</p>
          </div>
        </CardContent>
      </Card>
    </div>
  )
}
