"use client"

import { useState, useEffect } from 'react'
import { AdminLayout } from '@/components/admin/admin-layout'  // ✅ Correct - named import
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Label } from '@/components/ui/label'
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card'
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs'
import { Switch } from '@/components/ui/switch'
import { CreditCard, DollarSign, Shield } from 'lucide-react'
import { toast } from 'sonner'

interface PaymentSettings {
  stripePublishableKey: string
  stripeSecretKey: string
  paypalClientId: string
  paypalClientSecret: string
  paypalSandbox: boolean
  squareApplicationId: string
  squareAccessToken: string
  squareSandbox: boolean
  enabledGateways: {
    stripe: boolean
    paypal: boolean
    square: boolean
  }
}

export default function PaymentSettingsPage() {
  const [settings, setSettings] = useState<PaymentSettings | null>(null)
  const [loading, setLoading] = useState(true)
  const [saving, setSaving] = useState(false)

  useEffect(() => {
    fetchPaymentSettings()
  }, [])

  const fetchPaymentSettings = async () => {
    try {
      const response = await fetch('/api/admin/payments', {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('adminToken') || 'admin-token-placeholder'}`
        }
      })
      if (response.ok) {
        const data = await response.json()
        setSettings(data)
      } else {
        toast.error('Failed to load payment settings')
      }
    } catch (error) {
      toast.error('Error loading payment settings')
    } finally {
      setLoading(false)
    }
  }

  const saveSettings = async () => {
    if (!settings) return
    
    setSaving(true)
    try {
      const response = await fetch('/api/admin/payments', {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${localStorage.getItem('adminToken') || 'admin-token-placeholder'}`
        },
        body: JSON.stringify(settings)
      })
      
      if (response.ok) {
        toast.success('Payment settings saved successfully')
      } else {
        toast.error('Failed to save payment settings')
      }
    } catch (error) {
      toast.error('Error saving payment settings')
    } finally {
      setSaving(false)
    }
  }

  if (loading) {
    return (
      <AdminLayout>
        <div className="flex items-center justify-center h-64">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-purple-400"></div>
        </div>
      </AdminLayout>
    )
  }

  if (!settings) {
    return (
      <AdminLayout>
        <div className="text-white">Failed to load payment settings</div>
      </AdminLayout>
    )
  }

  return (
    <AdminLayout>
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-2">
            <CreditCard className="w-6 h-6 text-purple-400" />
            <h1 className="text-2xl font-bold text-white">Payment Settings</h1>
          </div>
          <Button onClick={saveSettings} disabled={saving} className="bg-purple-600 hover:bg-purple-700">
            {saving ? 'Saving...' : 'Save Settings'}
          </Button>
        </div>

        <Tabs defaultValue="gateways" className="space-y-4">
          <TabsList className="bg-white/10">
            <TabsTrigger value="gateways" className="text-white data-[state=active]:bg-purple-600">
              <Shield className="w-4 h-4 mr-2" />
              Payment Gateways
            </TabsTrigger>
            <TabsTrigger value="stripe" className="text-white data-[state=active]:bg-purple-600">
              Stripe
            </TabsTrigger>
            <TabsTrigger value="paypal" className="text-white data-[state=active]:bg-purple-600">
              PayPal
            </TabsTrigger>
            <TabsTrigger value="square" className="text-white data-[state=active]:bg-purple-600">
              Square
            </TabsTrigger>
          </TabsList>

        <TabsContent value="gateways">
          <Card className="bg-white/10 border-white/20">
            <CardHeader>
              <CardTitle className="text-white">Enabled Payment Gateways</CardTitle>
              <CardDescription className="text-gray-300">
                Choose which payment methods to enable for your customers
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex items-center justify-between">
                <div>
                  <Label className="text-white font-medium">Stripe</Label>
                  <p className="text-sm text-gray-400">Credit cards, digital wallets</p>
                </div>
                <Switch
                  checked={settings.enabledGateways.stripe}
                  onCheckedChange={(checked) => setSettings({
                    ...settings,
                    enabledGateways: { ...settings.enabledGateways, stripe: checked }
                  })}
                />
              </div>
              
              <div className="flex items-center justify-between">
                <div>
                  <Label className="text-white font-medium">PayPal</Label>
                  <p className="text-sm text-gray-400">PayPal payments and Express Checkout</p>
                </div>
                <Switch
                  checked={settings.enabledGateways.paypal}
                  onCheckedChange={(checked) => setSettings({
                    ...settings,
                    enabledGateways: { ...settings.enabledGateways, paypal: checked }
                  })}
                />
              </div>
              
              <div className="flex items-center justify-between">
                <div>
                  <Label className="text-white font-medium">Square</Label>
                  <p className="text-sm text-gray-400">Square payment processing</p>
                </div>
                <Switch
                  checked={settings.enabledGateways.square}
                  onCheckedChange={(checked) => setSettings({
                    ...settings,
                    enabledGateways: { ...settings.enabledGateways, square: checked }
                  })}
                />
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="stripe">
          <Card className="bg-white/10 border-white/20">
            <CardHeader>
              <CardTitle className="text-white">Stripe Configuration</CardTitle>
              <CardDescription className="text-gray-300">
                Configure your Stripe payment gateway settings
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div>
                <Label htmlFor="stripePublishableKey" className="text-white">Publishable Key</Label>
                <Input
                  id="stripePublishableKey"
                  value={settings.stripePublishableKey}
                  onChange={(e) => setSettings({...settings, stripePublishableKey: e.target.value})}
                  placeholder="pk_test_..."
                  className="bg-white/10 border-white/20 text-white"
                />
              </div>
              <div>
                <Label htmlFor="stripeSecretKey" className="text-white">Secret Key</Label>
                <Input
                  id="stripeSecretKey"
                  type="password"
                  value={settings.stripeSecretKey}
                  onChange={(e) => setSettings({...settings, stripeSecretKey: e.target.value})}
                  placeholder="sk_test_..."
                  className="bg-white/10 border-white/20 text-white"
                />
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="paypal">
          <Card className="bg-white/10 border-white/20">
            <CardHeader>
              <CardTitle className="text-white">PayPal Configuration</CardTitle>
              <CardDescription className="text-gray-300">
                Configure your PayPal payment gateway settings
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div>
                <Label htmlFor="paypalClientId" className="text-white">Client ID</Label>
                <Input
                  id="paypalClientId"
                  value={settings.paypalClientId}
                  onChange={(e) => setSettings({...settings, paypalClientId: e.target.value})}
                  className="bg-white/10 border-white/20 text-white"
                />
              </div>
              <div>
                <Label htmlFor="paypalClientSecret" className="text-white">Client Secret</Label>
                <Input
                  id="paypalClientSecret"
                  type="password"
                  value={settings.paypalClientSecret}
                  onChange={(e) => setSettings({...settings, paypalClientSecret: e.target.value})}
                  className="bg-white/10 border-white/20 text-white"
                />
              </div>
              <div className="flex items-center space-x-2">
                <Switch
                  id="paypalSandbox"
                  checked={settings.paypalSandbox}
                  onCheckedChange={(checked) => setSettings({...settings, paypalSandbox: checked})}
                />
                <Label htmlFor="paypalSandbox" className="text-white">Sandbox Mode</Label>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="square">
          <Card className="bg-white/10 border-white/20">
            <CardHeader>
              <CardTitle className="text-white">Square Configuration</CardTitle>
              <CardDescription className="text-gray-300">
                Configure your Square payment gateway settings
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div>
                <Label htmlFor="squareApplicationId" className="text-white">Application ID</Label>
                <Input
                  id="squareApplicationId"
                  value={settings.squareApplicationId}
                  onChange={(e) => setSettings({...settings, squareApplicationId: e.target.value})}
                  className="bg-white/10 border-white/20 text-white"
                />
              </div>
              <div>
                <Label htmlFor="squareAccessToken" className="text-white">Access Token</Label>
                <Input
                  id="squareAccessToken"
                  type="password"
                  value={settings.squareAccessToken}
                  onChange={(e) => setSettings({...settings, squareAccessToken: e.target.value})}
                  className="bg-white/10 border-white/20 text-white"
                />
              </div>
              <div className="flex items-center space-x-2">
                <Switch
                  id="squareSandbox"
                  checked={settings.squareSandbox}
                  onCheckedChange={(checked) => setSettings({...settings, squareSandbox: checked})}
                />
                <Label htmlFor="squareSandbox" className="text-white">Sandbox Mode</Label>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>

      <div className="flex justify-end">
        <Button
          onClick={saveSettings}
          disabled={saving}
          className="bg-green-600 hover:bg-green-700"
        >
          {saving ? (
            <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
          ) : null}
          Save Settings
        </Button>
      </div>
    </div>
    </AdminLayout>
  )
}