
import { NextRequest, NextResponse } from 'next/server'
import { z } from 'zod'
import { ContentDB } from '@/lib/database'

// Validation schemas
const HeroContentSchema = z.object({
  title: z.string().min(1),
  subtitle: z.string().min(1),
  ctaText: z.string().min(1),
  ctaLink: z.string().url(),
  backgroundVideo: z.string().optional(),
  backgroundImage: z.string().optional(),
  stats: z.object({
    experience: z.object({ number: z.string(), label: z.string() }),
    clients: z.object({ number: z.string(), label: z.string() }),
    support: z.object({ number: z.string(), label: z.string() })
  })
})

const AboutContentSchema = z.object({
  title: z.string().min(1),
  subtitle: z.string().min(1),
  description: z.string().min(1),
  mission: z.string().min(1),
  vision: z.string().min(1),
  values: z.array(z.object({
    title: z.string(),
    description: z.string(),
    icon: z.string()
  })),
  achievements: z.array(z.object({
    number: z.string(),
    label: z.string(),
    description: z.string()
  }))
})

const ServiceSchema = z.object({
  title: z.string().min(1),
  description: z.string().min(1),
  icon: z.string(),
  features: z.array(z.string()),
  price: z.string().optional(),
  category: z.string()
})

const TeamMemberSchema = z.object({
  name: z.string().min(1),
  position: z.string().min(1),
  bio: z.string(),
  image: z.string().optional(),
  linkedin: z.string().optional(),
  email: z.string().email().optional()
})

const ContactInfoSchema = z.object({
  address: z.string().min(1),
  phone: z.string().min(1),
  email: z.string().email(),
  hours: z.string(),
  socialMedia: z.object({
    facebook: z.string().optional(),
    twitter: z.string().optional(),
    linkedin: z.string().optional(),
    instagram: z.string().optional()
  })
})

// GET - Fetch content by type
export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const contentType = searchParams.get('type')

    // Verify admin authentication
    const isAuthenticated = await verifyAdminAuth(request)
    if (!isAuthenticated) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
    }

    switch (contentType) {
      case 'hero':
        const heroContent = await ContentDB.hero.get()
        return NextResponse.json(heroContent)

      case 'about':
        const aboutContent = await ContentDB.about.get()
        return NextResponse.json(aboutContent)

      case 'services':
        const services = await ContentDB.services.get()
        return NextResponse.json(services)

      case 'team':
        const team = await ContentDB.team.get()
        return NextResponse.json(team)

      case 'contact':
        const contact = await ContentDB.contact.get()
        return NextResponse.json(contact)

      default:
        return NextResponse.json({ error: 'Invalid content type' }, { status: 400 })
    }
  } catch (error) {
    console.error('Content fetch error:', error)
    return NextResponse.json({ error: 'Internal server error' }, { status: 500 })
  }
}

// POST - Create new content
export async function POST(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    let contentType = searchParams.get('type')
    const body = await request.json()

    // Allow both payload shapes: {type, data} or raw schema
    const payloadType = (body && typeof body === 'object' && 'type' in body) ? (body.type as string) : undefined
    const payloadData = (body && typeof body === 'object' && 'data' in body) ? (body.data as any) : body
    if (!contentType && payloadType) contentType = payloadType

    // Verify admin authentication
    const isAuthenticated = await verifyAdminAuth(request)
    if (!isAuthenticated) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
    }

    switch (contentType) {
      case 'hero':
        const validatedHero = HeroContentSchema.parse(payloadData)
        const newHero = await ContentDB.hero.create(validatedHero)
        return NextResponse.json(newHero, { status: 201 })

      case 'about':
        const validatedAbout = AboutContentSchema.parse(payloadData)
        const newAbout = await ContentDB.about.create(validatedAbout)
        return NextResponse.json(newAbout, { status: 201 })

      case 'services':
        const validatedService = ServiceSchema.parse(payloadData)
        const newService = await ContentDB.services.create(validatedService)
        return NextResponse.json(newService, { status: 201 })

      case 'team':
        const validatedTeamMember = TeamMemberSchema.parse(payloadData)
        const newTeamMember = await ContentDB.team.create({
          ...validatedTeamMember,
          image: validatedTeamMember.image || '/team.png',
        })
        return NextResponse.json(newTeamMember, { status: 201 })

      case 'contact':
        const validatedContact = ContactInfoSchema.parse(payloadData)
        const newContact = await ContentDB.contact.create(validatedContact)
        return NextResponse.json(newContact, { status: 201 })

      default:
        return NextResponse.json({ error: 'Invalid content type' }, { status: 400 })
    }
  } catch (error) {
    if (error instanceof z.ZodError) {
      return NextResponse.json({ error: 'Validation failed', details: error.errors }, { status: 400 })
    }
    console.error('Content creation error:', error)
    return NextResponse.json({ error: 'Internal server error' }, { status:  500 })
  }
}

// PUT - Update existing content
export async function PUT(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    let contentType = searchParams.get('type')
    const contentId = searchParams.get('id')
    const body = await request.json()

    // Allow both payload shapes: {type, data} or raw schema
    const payloadType = (body && typeof body === 'object' && 'type' in body) ? (body.type as string) : undefined
    const payloadData = (body && typeof body === 'object' && 'data' in body) ? (body.data as any) : body
    if (!contentType && payloadType) contentType = payloadType

    // Verify admin authentication
    const isAuthenticated = await verifyAdminAuth(request)
    if (!isAuthenticated) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
    }

    if (!contentId) {
      return NextResponse.json({ error: 'Content ID required' }, { status: 400 })
    }

    switch (contentType) {
      case 'hero':
        const validatedHero = HeroContentSchema.parse(payloadData)
        const updatedHero = await ContentDB.hero.update(contentId, validatedHero)
        return NextResponse.json(updatedHero)

      case 'about':
        const validatedAbout = AboutContentSchema.parse(payloadData)
        const updatedAbout = await ContentDB.about.update(contentId, validatedAbout)
        return NextResponse.json(updatedAbout)

      case 'services':
        const validatedService = ServiceSchema.parse(payloadData)
        const updatedService = await ContentDB.services.update(contentId, validatedService)
        return NextResponse.json(updatedService)

      case 'team':
        const validatedTeamMember = TeamMemberSchema.parse(payloadData)
        const updatedTeamMember = await ContentDB.team.update(contentId, {
          ...validatedTeamMember,
          image: validatedTeamMember.image || '/team.png',
        })
        return NextResponse.json(updatedTeamMember)

      case 'contact':
        const validatedContact = ContactInfoSchema.parse(payloadData)
        const updatedContact = await ContentDB.contact.update(contentId, validatedContact)
        return NextResponse.json(updatedContact)

      default:
        return NextResponse.json({ error: 'Invalid content type' }, { status: 400 })
    }
  } catch (error) {
    if (error instanceof z.ZodError) {
      return NextResponse.json({ error: 'Validation failed', details: error.errors }, { status: 400 })
    }
    console.error('Content update error:', error)
    return NextResponse.json({ error: 'Internal server error' }, { status: 500 })
  }
}

// DELETE - Remove content
export async function DELETE(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const contentType = searchParams.get('type')
    const contentId = searchParams.get('id')

    // Verify admin authentication
    const isAuthenticated = await verifyAdminAuth(request)
    if (!isAuthenticated) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
    }

    if (!contentType || !(contentType in ContentDB)) {
      return NextResponse.json({ error: 'Invalid content type' }, { status: 400 })
    }

    if (!contentId) {
      return NextResponse.json({ error: 'Content ID required' }, { status: 400 })
    }

    // @ts-ignore dynamic index
    await ContentDB[contentType].delete(contentId)
    return NextResponse.json({ success: true })
  } catch (error) {
    console.error('Content deletion error:', error)
    return NextResponse.json({ error: 'Internal server error' }, { status: 500 })
  }
}

// Helper function for admin authentication
async function verifyAdminAuth(request: NextRequest): Promise<boolean> {
  try {
    // Check for authorization header
    const authHeader = request.headers.get('authorization')
    if (!authHeader || !authHeader.startsWith('Bearer ')) {
      return false
    }

    // For now, we'll use a simple token check
    // In production, implement proper JWT verification
    const token = authHeader.substring(7)
    
    // Simple token validation (replace with proper JWT verification)
    if (token === process.env.ADMIN_TOKEN || token === 'admin-token-placeholder') {
      return true
    }

    return false
  } catch (error) {
    console.error('Auth verification error:', error)
    return false
  }
}
