import nodemailer from 'nodemailer'
import { NextRequest, NextResponse } from 'next/server'

export async function POST(request: NextRequest) {
  try {
    const { to, subject, content, adminSettings } = await request.json()
    
    console.log('Email request received:', { to, subject, hasAdminSettings: !!adminSettings })
    
    // Use admin settings if provided, otherwise fallback to environment variables
    const smtpHost = adminSettings?.email?.smtpHost || process.env.SMTP_HOST || 'smtp.gmail.com'
    const smtpPort = parseInt(adminSettings?.email?.smtpPort || process.env.SMTP_PORT || '587')
    const smtpUsername = adminSettings?.email?.smtpUsername || process.env.SMTP_USERNAME
    const smtpPassword = adminSettings?.email?.smtpPassword || process.env.SMTP_PASSWORD
    
    // Fix: Properly determine secure setting based on port
    const isSecure = smtpPort === 465
    
    const smtpConfig = {
      host: smtpHost,
      port: smtpPort,
      secure: isSecure, // true for 465, false for other ports like 587
      auth: {
        user: smtpUsername,
        pass: smtpPassword,
      },
      // Add these for better compatibility
      tls: {
        rejectUnauthorized: false // Accept self-signed certificates
      }
    }
    
    console.log('SMTP Config:', {
      host: smtpConfig.host,
      port: smtpConfig.port,
      secure: smtpConfig.secure,
      user: smtpConfig.auth.user ? '***configured***' : 'NOT SET',
      pass: smtpConfig.auth.pass ? '***configured***' : 'NOT SET'
    })
    
    const fromName = adminSettings?.email?.fromName || process.env.SMTP_FROM_NAME || 'Bishop & Knight Insurance'
    const fromEmail = adminSettings?.email?.fromEmail || process.env.SMTP_FROM_EMAIL || 'noreply@bishopknight.com'
    
    // Validate required fields
    if (!smtpConfig.auth.user || !smtpConfig.auth.pass) {
      console.error('SMTP credentials missing')
      return NextResponse.json({ 
        error: 'SMTP credentials not configured. Please set up email settings in admin panel or environment variables.',
        debug: {
          hasUser: !!smtpConfig.auth.user,
          hasPass: !!smtpConfig.auth.pass,
          envVarsSet: {
            SMTP_HOST: !!process.env.SMTP_HOST,
            SMTP_USERNAME: !!process.env.SMTP_USERNAME,
            SMTP_PASSWORD: !!process.env.SMTP_PASSWORD
          }
        }
      }, { status: 400 })
    }
    
    // FIXED: Use createTransport instead of createTransporter
    const transporter = nodemailer.createTransport(smtpConfig)
    
    console.log('Attempting to verify SMTP connection...')
    
    // Verify connection with timeout
    try {
      await Promise.race([
        transporter.verify(),
        new Promise((_, reject) => setTimeout(() => reject(new Error('Connection timeout')), 15000))
      ])
      console.log('SMTP connection verified successfully')
    } catch (verifyError) {
      console.error('SMTP verification failed:', verifyError)
      return NextResponse.json({ 
        error: 'SMTP connection failed',
        details: verifyError instanceof Error ? verifyError.message : 'Unknown verification error',
        config: {
          host: smtpConfig.host,
          port: smtpConfig.port,
          secure: smtpConfig.secure
        }
      }, { status: 500 })
    }
    
    console.log('Sending email...')
    
    await transporter.sendMail({
      from: `"${fromName}" <${fromEmail}>`,
      to,
      subject,
      text: content,
      html: content.replace(/\n/g, '<br>'),
    })
    
    console.log('Email sent successfully')
    return NextResponse.json({ success: true, message: 'Email sent successfully' })
    
  } catch (error) {
    console.error('Email sending error:', error)
    return NextResponse.json({ 
      error: 'Failed to send email', 
      details: error instanceof Error ? error.message : 'Unknown error',
      stack: error instanceof Error ? error.stack : undefined
    }, { status: 500 })
  }
}