"use client"

import { useState } from "react"
import { useRouter } from "next/navigation"
import { ChevronLeft, ChevronRight, Building2, Shield, CheckCircle, User, Phone, Mail, Copy } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Textarea } from "@/components/ui/textarea"
import { useShoppingCart } from "@/components/shopping-cart-provider"

interface CustomerInfo {
  firstName: string
  lastName: string
  email: string
  phone: string
  address: string
  city: string
  state: string
  zipCode: string
  dateOfBirth: string
  occupation: string
  specialRequests: string
}

interface PaymentInfo {
  cardNumber: string
  expiryDate: string
  cvv: string
  cardholderName: string
  billingAddress: string
  billingCity: string
  billingState: string
  billingZipCode: string
}

const steps = [
  { id: 1, name: "Customer Information", icon: User },
  { id: 2, name: "Bank Transfer Details", icon: Building2 },
  { id: 3, name: "Review & Confirm", icon: Shield },
  { id: 4, name: "Invoice Generated", icon: CheckCircle },
]

export default function CheckoutFlow() {
  const [currentStep, setCurrentStep] = useState(1)
  const [customerInfo, setCustomerInfo] = useState<CustomerInfo>({
    firstName: "",
    lastName: "",
    email: "",
    phone: "",
    address: "",
    city: "",
    state: "",
    zipCode: "",
    dateOfBirth: "",
    occupation: "",
    specialRequests: "",
  })
  const [paymentInfo, setPaymentInfo] = useState<PaymentInfo>({
    cardNumber: "",
    expiryDate: "",
    cvv: "",
    cardholderName: "",
    billingAddress: "",
    billingCity: "",
    billingState: "",
    billingZipCode: "",
  })
  const [orderNumber, setOrderNumber] = useState("")
  const [invoiceNumber, setInvoiceNumber] = useState("")
  const [isProcessing, setIsProcessing] = useState(false)

  const { cartItems, getTotalPrice, clearCart } = useShoppingCart()
  const router = useRouter()

  console.log("[v0] CheckoutFlow - cartItems:", cartItems)
  console.log("[v0] CheckoutFlow - cartItems.length:", cartItems.length)
  console.log("[v0] CheckoutFlow - currentStep:", currentStep)

  const bankDetails = {
    bankName: "First Bank of Nigeria",
    accountName: "Bishop & Knight Insurance Brokers Limited",
    accountNumber: "2034567890",
    sortCode: "011-152-003",
  }

  const formatPrice = (price: number) => {
    return new Intl.NumberFormat("en-NG", {
      style: "currency",
      currency: "NGN",
      minimumFractionDigits: 0,
    }).format(price)
  }

  const handleNext = () => {
    if (currentStep < 4) {
      setCurrentStep(currentStep + 1)
    }
  }

  const handlePrevious = () => {
    if (currentStep > 1) {
      setCurrentStep(currentStep - 1)
    }
  }

  const handleSubmitOrder = async () => {
    setIsProcessing(true)

    // Simulate invoice generation
    await new Promise((resolve) => setTimeout(resolve, 2000))

    // Generate invoice number
    const invoiceNum = `INV-BK-${Date.now().toString().slice(-8)}`
    setInvoiceNumber(invoiceNum)

    // Generate order number
    const orderNum = `BK${Date.now().toString().slice(-6)}`
    setOrderNumber(orderNum)

    setIsProcessing(false)
    setCurrentStep(4)

    // Simulate sending invoice email
    console.log(`[v0] Invoice ${invoiceNum} sent to ${customerInfo.email}`)
  }

  const copyToClipboard = (text: string) => {
    navigator.clipboard.writeText(text)
  }

  const nigerianStates = [
    "Abia",
    "Adamawa",
    "Akwa Ibom",
    "Anambra",
    "Bauchi",
    "Bayelsa",
    "Benue",
    "Borno",
    "Cross River",
    "Delta",
    "Ebonyi",
    "Edo",
    "Ekiti",
    "Enugu",
    "FCT",
    "Gombe",
    "Imo",
    "Jigawa",
    "Kaduna",
    "Kano",
    "Katsina",
    "Kebbi",
    "Kogi",
    "Kwara",
    "Lagos",
    "Nasarawa",
    "Niger",
    "Ogun",
    "Ondo",
    "Osun",
    "Oyo",
    "Plateau",
    "Rivers",
    "Sokoto",
    "Taraba",
    "Yobe",
    "Zamfara",
  ]

  if (cartItems.length === 0 && currentStep < 4) {
    console.log("[v0] Showing empty cart message - cartItems.length:", cartItems.length, "currentStep:", currentStep)
    return (
      <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="text-center py-16">
          <div className="bg-white rounded-3xl shadow-lg p-12">
            <div className="w-24 h-24 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-6">
              <Building2 className="w-12 h-12 text-gray-400" />
            </div>
            <h2 className="text-2xl font-bold text-gray-900 mb-4">Your cart is empty</h2>
            <p className="text-gray-600 mb-8">
              Add some insurance services to your cart before proceeding to checkout.
            </p>
            <Button
              onClick={() => router.push("/shop")}
              className="bg-gradient-to-r from-purple-600 to-blue-600 hover:from-purple-700 hover:to-blue-700 text-white px-8 py-3 rounded-full"
            >
              Browse Services
            </Button>
          </div>
        </div>
      </div>
    )
  }

  console.log("[v0] Rendering checkout flow with items:", cartItems.length)

  return (
    <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
      {/* Progress Steps */}
      <div className="mb-12">
        <div className="flex items-center justify-center">
          {steps.map((step, index) => {
            const IconComponent = step.icon
            const isActive = currentStep === step.id
            const isCompleted = currentStep > step.id

            return (
              <div key={step.id} className="flex items-center">
                <div
                  className={`flex items-center justify-center w-12 h-12 rounded-full border-2 transition-all duration-300 ${
                    isActive
                      ? "bg-gradient-to-r from-purple-600 to-blue-600 border-purple-600 text-white"
                      : isCompleted
                        ? "bg-green-500 border-green-500 text-white"
                        : "bg-white border-gray-300 text-gray-400"
                  }`}
                >
                  <IconComponent className="w-5 h-5" />
                </div>
                <div className="ml-3 hidden sm:block">
                  <p
                    className={`text-sm font-medium ${isActive ? "text-purple-600" : isCompleted ? "text-green-600" : "text-gray-500"}`}
                  >
                    {step.name}
                  </p>
                </div>
                {index < steps.length - 1 && (
                  <div className={`w-16 h-0.5 mx-4 ${isCompleted ? "bg-green-500" : "bg-gray-300"}`}></div>
                )}
              </div>
            )
          })}
        </div>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
        {/* Main Content */}
        <div className="lg:col-span-2">
          <div className="bg-white rounded-3xl shadow-lg p-8">
            {/* Step 1: Customer Information */}
            {currentStep === 1 && (
              <div>
                <div className="flex items-center mb-8">
                  <User className="w-6 h-6 text-purple-600 mr-3" />
                  <h2 className="text-2xl font-bold text-gray-900">Customer Information</h2>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div>
                    <Label htmlFor="firstName">First Name *</Label>
                    <Input
                      id="firstName"
                      value={customerInfo.firstName}
                      onChange={(e) => setCustomerInfo({ ...customerInfo, firstName: e.target.value })}
                      className="mt-1"
                      required
                    />
                  </div>
                  <div>
                    <Label htmlFor="lastName">Last Name *</Label>
                    <Input
                      id="lastName"
                      value={customerInfo.lastName}
                      onChange={(e) => setCustomerInfo({ ...customerInfo, lastName: e.target.value })}
                      className="mt-1"
                      required
                    />
                  </div>
                  <div>
                    <Label htmlFor="email">Email Address *</Label>
                    <Input
                      id="email"
                      type="email"
                      value={customerInfo.email}
                      onChange={(e) => setCustomerInfo({ ...customerInfo, email: e.target.value })}
                      className="mt-1"
                      required
                    />
                  </div>
                  <div>
                    <Label htmlFor="phone">Phone Number *</Label>
                    <Input
                      id="phone"
                      type="tel"
                      value={customerInfo.phone}
                      onChange={(e) => setCustomerInfo({ ...customerInfo, phone: e.target.value })}
                      className="mt-1"
                      placeholder="+234 xxx xxx xxxx"
                      required
                    />
                  </div>
                  <div className="md:col-span-2">
                    <Label htmlFor="address">Address *</Label>
                    <Input
                      id="address"
                      value={customerInfo.address}
                      onChange={(e) => setCustomerInfo({ ...customerInfo, address: e.target.value })}
                      className="mt-1"
                      required
                    />
                  </div>
                  <div>
                    <Label htmlFor="city">City *</Label>
                    <Input
                      id="city"
                      value={customerInfo.city}
                      onChange={(e) => setCustomerInfo({ ...customerInfo, city: e.target.value })}
                      className="mt-1"
                      required
                    />
                  </div>
                  <div>
                    <Label htmlFor="state">State *</Label>
                    <select
                      id="state"
                      value={customerInfo.state}
                      onChange={(e) => setCustomerInfo({ ...customerInfo, state: e.target.value })}
                      className="mt-1 w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-purple-500"
                      required
                    >
                      <option value="">Select State</option>
                      {nigerianStates.map((state) => (
                        <option key={state} value={state}>
                          {state}
                        </option>
                      ))}
                    </select>
                  </div>
                  <div>
                    <Label htmlFor="dateOfBirth">Date of Birth *</Label>
                    <Input
                      id="dateOfBirth"
                      type="date"
                      value={customerInfo.dateOfBirth}
                      onChange={(e) => setCustomerInfo({ ...customerInfo, dateOfBirth: e.target.value })}
                      className="mt-1"
                      required
                    />
                  </div>
                  <div>
                    <Label htmlFor="occupation">Occupation *</Label>
                    <Input
                      id="occupation"
                      value={customerInfo.occupation}
                      onChange={(e) => setCustomerInfo({ ...customerInfo, occupation: e.target.value })}
                      className="mt-1"
                      required
                    />
                  </div>
                  <div className="md:col-span-2">
                    <Label htmlFor="specialRequests">Special Requests (Optional)</Label>
                    <Textarea
                      id="specialRequests"
                      value={customerInfo.specialRequests}
                      onChange={(e) => setCustomerInfo({ ...customerInfo, specialRequests: e.target.value })}
                      className="mt-1"
                      rows={3}
                      placeholder="Any special requirements or additional information..."
                    />
                  </div>
                </div>
              </div>
            )}

            {/* Step 2: Bank Transfer Details */}
            {currentStep === 2 && (
              <div>
                <div className="flex items-center mb-8">
                  <Building2 className="w-6 h-6 text-purple-600 mr-3" />
                  <h2 className="text-2xl font-bold text-gray-900">Bank Transfer Details</h2>
                </div>

                <div className="space-y-6">
                  <div className="bg-gradient-to-r from-green-50 to-blue-50 p-6 rounded-xl border border-green-200">
                    <div className="flex items-center mb-4">
                      <Shield className="w-5 h-5 text-green-600 mr-2" />
                      <span className="text-sm font-medium text-green-800">Secure Bank Transfer</span>
                    </div>
                    <p className="text-sm text-green-700">
                      Make your payment directly to our bank account. Your invoice will be sent to your email with
                      payment instructions.
                    </p>
                  </div>

                  <div className="bg-white border-2 border-purple-200 rounded-xl p-6">
                    <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
                      <Building2 className="w-5 h-5 mr-2 text-purple-600" />
                      Bank Account Details
                    </h3>

                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      <div className="space-y-3">
                        <div>
                          <Label className="text-sm font-medium text-gray-600">Bank Name</Label>
                          <div className="flex items-center justify-between bg-gray-50 p-3 rounded-lg mt-1">
                            <span className="font-semibold text-gray-900">{bankDetails.bankName}</span>
                            <Button
                              variant="ghost"
                              size="sm"
                              onClick={() => copyToClipboard(bankDetails.bankName)}
                              className="p-1 h-auto"
                            >
                              <Copy className="w-4 h-4" />
                            </Button>
                          </div>
                        </div>

                        <div>
                          <Label className="text-sm font-medium text-gray-600">Account Name</Label>
                          <div className="flex items-center justify-between bg-gray-50 p-3 rounded-lg mt-1">
                            <span className="font-semibold text-gray-900">{bankDetails.accountName}</span>
                            <Button
                              variant="ghost"
                              size="sm"
                              onClick={() => copyToClipboard(bankDetails.accountName)}
                              className="p-1 h-auto"
                            >
                              <Copy className="w-4 h-4" />
                            </Button>
                          </div>
                        </div>
                      </div>

                      <div className="space-y-3">
                        <div>
                          <Label className="text-sm font-medium text-gray-600">Account Number</Label>
                          <div className="flex items-center justify-between bg-gray-50 p-3 rounded-lg mt-1">
                            <span className="font-semibold text-gray-900 text-lg">{bankDetails.accountNumber}</span>
                            <Button
                              variant="ghost"
                              size="sm"
                              onClick={() => copyToClipboard(bankDetails.accountNumber)}
                              className="p-1 h-auto"
                            >
                              <Copy className="w-4 h-4" />
                            </Button>
                          </div>
                        </div>

                        <div>
                          <Label className="text-sm font-medium text-gray-600">Sort Code</Label>
                          <div className="flex items-center justify-between bg-gray-50 p-3 rounded-lg mt-1">
                            <span className="font-semibold text-gray-900">{bankDetails.sortCode}</span>
                            <Button
                              variant="ghost"
                              size="sm"
                              onClick={() => copyToClipboard(bankDetails.sortCode)}
                              className="p-1 h-auto"
                            >
                              <Copy className="w-4 h-4" />
                            </Button>
                          </div>
                        </div>
                      </div>
                    </div>
                  </div>

                  <div className="bg-amber-50 border border-amber-200 rounded-xl p-6">
                    <h4 className="font-semibold text-amber-800 mb-2">Important Payment Instructions:</h4>
                    <ul className="text-sm text-amber-700 space-y-1">
                      <li>• Use your invoice number as payment description/reference</li>
                      <li>• Keep your payment receipt for records</li>
                      <li>• Policy processing begins after payment confirmation</li>
                      <li>• Contact us if you need assistance: +234 803 123 4567</li>
                    </ul>
                  </div>
                </div>
              </div>
            )}

            {/* Step 3: Review & Confirm */}
            {currentStep === 3 && (
              <div>
                <div className="flex items-center mb-8">
                  <Shield className="w-6 h-6 text-purple-600 mr-3" />
                  <h2 className="text-2xl font-bold text-gray-900">Review & Confirm</h2>
                </div>

                <div className="space-y-8">
                  {/* Customer Information Review */}
                  <div className="bg-gray-50 p-6 rounded-xl">
                    <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
                      <User className="w-5 h-5 mr-2" />
                      Customer Information
                    </h3>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                      <div>
                        <span className="font-medium">Name:</span> {customerInfo.firstName} {customerInfo.lastName}
                      </div>
                      <div>
                        <span className="font-medium">Email:</span> {customerInfo.email}
                      </div>
                      <div>
                        <span className="font-medium">Phone:</span> {customerInfo.phone}
                      </div>
                      <div>
                        <span className="font-medium">Occupation:</span> {customerInfo.occupation}
                      </div>
                      <div className="md:col-span-2">
                        <span className="font-medium">Address:</span> {customerInfo.address}, {customerInfo.city},{" "}
                        {customerInfo.state}
                      </div>
                    </div>
                  </div>

                  {/* Payment Information Review */}
                  <div className="bg-gray-50 p-6 rounded-xl">
                    <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
                      <Building2 className="w-5 h-5 mr-2 text-purple-600" />
                      Payment Information
                    </h3>
                    <div className="text-sm">
                      <div>
                        <span className="font-medium">Bank:</span> {bankDetails.bankName}
                      </div>
                      <div>
                        <span className="font-medium">Account Name:</span> {bankDetails.accountName}
                      </div>
                      <div>
                        <span className="font-medium">Account Number:</span> {bankDetails.accountNumber}
                      </div>
                      <div>
                        <span className="font-medium">Sort Code:</span> {bankDetails.sortCode}
                      </div>
                    </div>
                  </div>

                  {/* Order Items Review */}
                  <div className="bg-gray-50 p-6 rounded-xl">
                    <h3 className="text-lg font-semibold text-gray-900 mb-4">Order Items</h3>
                    <div className="space-y-4">
                      {cartItems.map((item) => (
                        <div
                          key={item.id}
                          className="flex items-center justify-between py-3 border-b border-gray-200 last:border-b-0"
                        >
                          <div className="flex items-center gap-4">
                            <img
                              src={item.image || "/placeholder.svg"}
                              alt={item.name}
                              className="w-16 h-16 object-cover rounded-lg"
                            />
                            <div>
                              <h4 className="font-semibold text-gray-900">{item.name}</h4>
                              <p className="text-sm text-gray-600">Quantity: {item.quantity}</p>
                            </div>
                          </div>
                          <div className="text-right">
                            <div className="font-semibold text-gray-900">{formatPrice(item.price * item.quantity)}</div>
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>
                </div>
              </div>
            )}

            {/* Step 4: Invoice Generated */}
            {currentStep === 4 && (
              <div className="text-center py-12">
                <div className="w-24 h-24 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-6">
                  <CheckCircle className="w-12 h-12 text-green-600" />
                </div>
                <h2 className="text-3xl font-bold text-gray-900 mb-4">Invoice Generated!</h2>
                <p className="text-lg text-gray-600 mb-6">Your invoice has been sent to your email address</p>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-8">
                  <div className="bg-gradient-to-r from-purple-50 to-blue-50 p-6 rounded-xl border border-purple-200">
                    <div className="text-sm text-gray-600 mb-2">Invoice Number</div>
                    <div className="text-xl font-bold text-purple-600 flex items-center justify-center gap-2">
                      {invoiceNumber}
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => copyToClipboard(invoiceNumber)}
                        className="p-1 h-auto"
                      >
                        <Copy className="w-4 h-4" />
                      </Button>
                    </div>
                  </div>

                  <div className="bg-gradient-to-r from-green-50 to-teal-50 p-6 rounded-xl border border-green-200">
                    <div className="text-sm text-gray-600 mb-2">Order Number</div>
                    <div className="text-xl font-bold text-green-600">{orderNumber}</div>
                  </div>
                </div>

                <div className="bg-amber-50 border border-amber-200 rounded-xl p-6 mb-8">
                  <h3 className="font-semibold text-amber-800 mb-3">Next Steps:</h3>
                  <div className="text-left space-y-2 text-sm text-amber-700">
                    <div className="flex items-start gap-2">
                      <span className="font-semibold">1.</span>
                      <span>Transfer the total amount to our bank account using the details provided</span>
                    </div>
                    <div className="flex items-start gap-2">
                      <span className="font-semibold">2.</span>
                      <span>
                        Use invoice number <strong>{invoiceNumber}</strong> as your payment description
                      </span>
                    </div>
                    <div className="flex items-start gap-2">
                      <span className="font-semibold">3.</span>
                      <span>Keep your payment receipt for verification purposes</span>
                    </div>
                    <div className="flex items-start gap-2">
                      <span className="font-semibold">4.</span>
                      <span>We'll process your policy within 24 hours of payment confirmation</span>
                    </div>
                  </div>
                </div>

                <div className="space-y-4 text-left max-w-md mx-auto mb-8">
                  <div className="flex items-center text-gray-600">
                    <Mail className="w-5 h-5 mr-3 text-purple-600" />
                    Invoice sent to {customerInfo.email}
                  </div>
                  <div className="flex items-center text-gray-600">
                    <Phone className="w-5 h-5 mr-3 text-purple-600" />
                    Support: +234 803 123 4567
                  </div>
                  <div className="flex items-center text-gray-600">
                    <Shield className="w-5 h-5 mr-3 text-purple-600" />
                    Policy documents within 24 hours of payment
                  </div>
                </div>

                <div className="flex flex-col sm:flex-row gap-4 justify-center">
                  <Button
                    onClick={() => router.push("/shop")}
                    className="bg-gradient-to-r from-purple-600 to-blue-600 hover:from-purple-700 hover:to-blue-700 text-white px-8 py-3 rounded-full"
                  >
                    Continue Shopping
                  </Button>
                  <Button
                    variant="outline"
                    onClick={() => router.push("/")}
                    className="border-purple-200 text-purple-600 hover:bg-purple-50 px-8 py-3 rounded-full bg-transparent"
                  >
                    Back to Home
                  </Button>
                </div>
              </div>
            )}

            {/* Navigation Buttons */}
            {currentStep < 4 && (
              <div className="flex justify-between mt-12 pt-8 border-t border-gray-200">
                <Button
                  variant="outline"
                  onClick={handlePrevious}
                  disabled={currentStep === 1}
                  className="flex items-center gap-2 border-gray-300 text-gray-600 hover:bg-gray-50 bg-transparent"
                >
                  <ChevronLeft className="w-4 h-4" />
                  Previous
                </Button>

                {currentStep === 3 ? (
                  <Button
                    onClick={handleSubmitOrder}
                    disabled={isProcessing}
                    className="flex items-center gap-2 bg-gradient-to-r from-purple-600 to-blue-600 hover:from-purple-700 hover:to-blue-700 text-white px-8 py-3 rounded-full"
                  >
                    {isProcessing ? (
                      <>
                        <div className="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                        Processing...
                      </>
                    ) : (
                      <>
                        Complete Order
                        <ChevronRight className="w-4 h-4" />
                      </>
                    )}
                  </Button>
                ) : (
                  <Button
                    onClick={handleNext}
                    className="flex items-center gap-2 bg-gradient-to-r from-purple-600 to-blue-600 hover:from-purple-700 hover:to-blue-700 text-white px-8 py-3 rounded-full"
                  >
                    Next
                    <ChevronRight className="w-4 h-4" />
                  </Button>
                )}
              </div>
            )}
          </div>
        </div>

        {/* Order Summary Sidebar */}
        {currentStep < 4 && (
          <div className="lg:col-span-1">
            <div className="bg-white rounded-3xl shadow-lg p-6 sticky top-8">
              <h3 className="text-xl font-bold text-gray-900 mb-6">Order Summary</h3>

              <div className="space-y-4 mb-6">
                {cartItems.map((item) => (
                  <div key={item.id} className="flex items-center gap-3">
                    <img
                      src={item.image || "/placeholder.svg"}
                      alt={item.name}
                      className="w-12 h-12 object-cover rounded-lg"
                    />
                    <div className="flex-1">
                      <h4 className="font-medium text-gray-900 text-sm">{item.name}</h4>
                      <p className="text-xs text-gray-600">Qty: {item.quantity}</p>
                    </div>
                    <div className="text-sm font-semibold text-gray-900">{formatPrice(item.price * item.quantity)}</div>
                  </div>
                ))}
              </div>

              <div className="border-t border-gray-200 pt-4">
                <div className="flex justify-between items-center mb-2">
                  <span className="text-gray-600">Subtotal:</span>
                  <span className="font-semibold">{formatPrice(getTotalPrice())}</span>
                </div>
                <div className="flex justify-between items-center mb-2">
                  <span className="text-gray-600">Processing Fee:</span>
                  <span className="font-semibold">{formatPrice(5000)}</span>
                </div>
                <div className="flex justify-between items-center text-lg font-bold text-gray-900 pt-2 border-t border-gray-200">
                  <span>Total:</span>
                  <span>{formatPrice(getTotalPrice() + 5000)}</span>
                </div>
              </div>

              <div className="mt-6 p-4 bg-gradient-to-r from-purple-50 to-blue-50 rounded-xl border border-purple-200">
                <div className="flex items-center mb-2">
                  <Shield className="w-4 h-4 text-purple-600 mr-2" />
                  <span className="text-sm font-medium text-purple-800">Secure Checkout</span>
                </div>
                <p className="text-xs text-purple-700">Your information is protected with 256-bit SSL encryption</p>
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  )
}
