"use client"

import { Card, CardContent } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Textarea } from "@/components/ui/textarea"
import { MapPin, Phone, Mail, Clock, Send, MessageCircle, Zap } from "lucide-react"
import { useEffect, useState } from "react"

export function ContactSection() {
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 },
    )

    const element = document.getElementById("contact")
    if (element) observer.observe(element)

    return () => observer.disconnect()
  }, [])

  return (
    <section
      id="contact"
      className="py-20 bg-gradient-to-br from-white via-purple-50/20 to-blue-50/20 relative overflow-hidden"
    >
      {/* Background Elements */}
      <div className="absolute inset-0 overflow-hidden">
        <div className="absolute top-20 left-20 w-72 h-72 bg-gradient-to-r from-purple-300/20 to-pink-300/20 rounded-full blur-3xl animate-blob"></div>
        <div className="absolute bottom-20 right-20 w-64 h-64 bg-gradient-to-r from-blue-300/20 to-cyan-300/20 rounded-full blur-3xl animate-blob animation-delay-2000"></div>
      </div>

      <div className="container mx-auto px-4 relative z-10">
        <div
          className={`text-center mb-16 transition-all duration-1000 ${isVisible ? "opacity-100 translate-y-0" : "opacity-0 translate-y-10"}`}
        >
          <div className="inline-flex items-center px-6 py-3 bg-gradient-to-r from-purple-100 to-blue-100 text-purple-800 rounded-full text-sm font-medium border border-purple-200/50 mb-6">
            <MessageCircle className="h-4 w-4 mr-2 animate-pulse" />
            Get In Touch
          </div>

          <h2 className="text-4xl lg:text-5xl font-bold mb-6">
            <span className="bg-gradient-to-r from-purple-600 to-blue-600 bg-clip-text text-transparent">
              Ready to Get
            </span>
            <br />
            <span className="text-slate-900">Protected?</span>
          </h2>

          <p className="text-xl text-slate-600 max-w-3xl mx-auto leading-relaxed">
            Experience instant quotes, personalized service, and expert guidance. Your protection is just one
            conversation away.
          </p>
        </div>

        <div className="grid lg:grid-cols-2 gap-12">
          {/* Contact Form */}
          <Card
            className={`shadow-2xl border-0 bg-white/80 backdrop-blur-sm transition-all duration-1000 hover:shadow-3xl ${isVisible ? "opacity-100 translate-x-0" : "opacity-0 -translate-x-10"}`}
          >
            <CardContent className="p-8">
              <div className="flex items-center mb-6">
                <div className="p-3 bg-gradient-to-r from-purple-500 to-blue-500 rounded-xl mr-4">
                  <Send className="h-6 w-6 text-white" />
                </div>
                <h3 className="text-2xl font-bold text-slate-900">Send Us a Message</h3>
              </div>

              <form className="space-y-6">
                <div className="grid md:grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-semibold text-slate-700 mb-2">First Name</label>
                    <Input
                      placeholder="Your first name"
                      className="border-slate-300 focus:border-purple-500 focus:ring-purple-500 bg-white/50 backdrop-blur-sm"
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-semibold text-slate-700 mb-2">Last Name</label>
                    <Input
                      placeholder="Your last name"
                      className="border-slate-300 focus:border-purple-500 focus:ring-purple-500 bg-white/50 backdrop-blur-sm"
                    />
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-semibold text-slate-700 mb-2">Email</label>
                  <Input
                    type="email"
                    placeholder="your.email@example.com"
                    className="border-slate-300 focus:border-purple-500 focus:ring-purple-500 bg-white/50 backdrop-blur-sm"
                  />
                </div>

                <div>
                  <label className="block text-sm font-semibold text-slate-700 mb-2">Phone</label>
                  <Input
                    type="tel"
                    placeholder="+234 801 234 5678"
                    className="border-slate-300 focus:border-purple-500 focus:ring-purple-500 bg-white/50 backdrop-blur-sm"
                  />
                </div>

                <div>
                  <label className="block text-sm font-semibold text-slate-700 mb-2">Insurance Type</label>
                  <select className="w-full p-3 border border-slate-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-purple-500 bg-white/50 backdrop-blur-sm">
                    <option>Select insurance type</option>
                    <option>Smart Motor Insurance</option>
                    <option>Digital Health & Life</option>
                    <option>Property Protection Plus</option>
                    <option>Marine & Aviation Pro</option>
                    <option>Professional Shield</option>
                    <option>Employee Benefits 360</option>
                    <option>Custom Solution</option>
                  </select>
                </div>

                <div>
                  <label className="block text-sm font-semibold text-slate-700 mb-2">Message</label>
                  <Textarea
                    placeholder="Tell us about your insurance needs and we'll create a personalized solution..."
                    className="border-slate-300 focus:border-purple-500 focus:ring-purple-500 min-h-[120px] bg-white/50 backdrop-blur-sm"
                  />
                </div>

                <Button className="w-full bg-gradient-to-r from-purple-600 to-blue-600 hover:from-purple-700 hover:to-blue-700 text-white py-4 text-lg rounded-full group transition-all duration-300">
                  <span className="flex items-center justify-center">
                    <Send className="mr-2 h-5 w-5 group-hover:translate-x-1 transition-transform duration-300" />
                    Send Message & Get Quote
                  </span>
                </Button>
              </form>
            </CardContent>
          </Card>

          {/* Contact Information */}
          <div
            className={`space-y-8 transition-all duration-1000 delay-300 ${isVisible ? "opacity-100 translate-x-0" : "opacity-0 translate-x-10"}`}
          >
            <div>
              <h3 className="text-2xl font-bold text-slate-900 mb-8">Multiple Ways to Connect</h3>

              <div className="space-y-6">
                {[
                  {
                    icon: MapPin,
                    title: "Visit Our Office",
                    content: "123 Victoria Island, Lagos\nLagos State, Nigeria",
                    gradient: "from-blue-500 to-cyan-500",
                    action: "Get Directions",
                  },
                  {
                    icon: Phone,
                    title: "Call Us Now",
                    content: "+234 801 234 5678",
                    gradient: "from-green-500 to-teal-500",
                    action: "Call Now",
                    href: "tel:+2348012345678",
                  },
                  {
                    icon: Mail,
                    title: "Email Us",
                    content: "info@bishopknight.ng",
                    gradient: "from-orange-500 to-red-500",
                    action: "Send Email",
                    href: "mailto:info@bishopknight.ng",
                  },
                  {
                    icon: Clock,
                    title: "Business Hours",
                    content: "Monday - Friday: 8:00am - 5:00pm\nSaturday: 9:00am - 2:00pm\nSunday: Closed",
                    gradient: "from-purple-500 to-pink-500",
                    action: "24/7 Online",
                  },
                ].map((item, index) => (
                  <Card
                    key={index}
                    className="group hover:shadow-xl transition-all duration-300 border-0 bg-white/60 backdrop-blur-sm hover:-translate-y-1"
                  >
                    <CardContent className="p-6">
                      <div className="flex items-start space-x-4">
                        <div
                          className={`p-3 bg-gradient-to-r ${item.gradient} rounded-xl group-hover:scale-110 transition-transform duration-300`}
                        >
                          <item.icon className="h-6 w-6 text-white" />
                        </div>
                        <div className="flex-1">
                          <h4 className="font-bold text-slate-900 mb-2">{item.title}</h4>
                          <p className="text-slate-600 whitespace-pre-line mb-3">{item.content}</p>
                          {item.href ? (
                            <a
                              href={item.href}
                              className={`inline-flex items-center text-sm font-semibold bg-gradient-to-r ${item.gradient} bg-clip-text text-transparent hover:scale-105 transition-transform duration-300`}
                            >
                              {item.action}
                              <Zap className="ml-1 h-4 w-4" />
                            </a>
                          ) : (
                            <span
                              className={`inline-flex items-center text-sm font-semibold bg-gradient-to-r ${item.gradient} bg-clip-text text-transparent`}
                            >
                              {item.action}
                              <Zap className="ml-1 h-4 w-4" />
                            </span>
                          )}
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </div>

            {/* CTA Card */}
            <Card className="bg-gradient-to-r from-purple-600 via-blue-600 to-cyan-600 text-white border-0 shadow-2xl overflow-hidden relative group">
              <div className="absolute inset-0 bg-gradient-to-r from-white/10 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>
              <CardContent className="p-8 text-center relative z-10">
                <div className="p-4 bg-white/20 backdrop-blur-sm rounded-2xl inline-block mb-4">
                  <Zap className="h-8 w-8 text-white" />
                </div>
                <h4 className="text-2xl font-bold mb-4">Get Instant Quote</h4>
                <p className="text-blue-100 mb-6 leading-relaxed">
                  Experience AI-powered insurance matching. Get personalized quotes in under 60 seconds with our smart
                  recommendation engine.
                </p>
                <Button
                  variant="secondary"
                  className="bg-white text-purple-600 hover:bg-slate-100 px-8 py-3 rounded-full font-semibold group transition-all duration-300"
                >
                  <Zap className="mr-2 h-5 w-5 group-hover:animate-pulse" />
                  Start Smart Quote
                </Button>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </section>
  )
}
