"use client"

import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Menu, X, Phone, Mail, ShoppingBag } from "lucide-react"
import Link from "next/link"

export function Header() {
  const [isMenuOpen, setIsMenuOpen] = useState(false)
  const [isScrolled, setIsScrolled] = useState(false)

  useEffect(() => {
    const handleScroll = () => {
      setIsScrolled(window.scrollY > 50)
    }
    window.addEventListener("scroll", handleScroll)
    return () => window.removeEventListener("scroll", handleScroll)
  }, [])

  return (
    <header
      className={`fixed top-0 left-0 right-0 z-50 transition-all duration-500 ${
        isScrolled
          ? "bg-white/90 backdrop-blur-xl shadow-2xl border-b border-purple-100"
          : "bg-gradient-to-r from-purple-900/10 to-blue-900/10 backdrop-blur-sm"
      }`}
    >
      <div className="container mx-auto px-4">
        <div className="flex items-center justify-between h-20">
          <Link href="/" className="flex items-center space-x-3 group">
            <div className="relative">
              <img
                src="/logo.png"
                alt="Bishop & Knight Insurance Brokers"
                className="h-14 w-auto object-contain transition-transform duration-300 group-hover:scale-105"
              />
              <div className="absolute inset-0 bg-gradient-to-r from-purple-500/20 to-blue-500/20 rounded-lg opacity-0 group-hover:opacity-100 transition-opacity duration-300 blur-xl"></div>
            </div>
          </Link>

          <nav className="hidden lg:flex items-center space-x-1">
            {[
              { href: "/", label: "Home" },
              { href: "/about", label: "About" },
              { href: "/services", label: "Services" },
              { href: "/shop", label: "Shop" },
              { href: "/leadership", label: "Leadership" },
              { href: "/contact", label: "Contact" },
            ].map((item) => (
              <Link
                key={item.href}
                href={item.href}
                className="relative px-4 py-2 text-slate-700 hover:text-purple-600 transition-all duration-300 group"
              >
                <span className="relative z-10">{item.label}</span>
                <div className="absolute inset-0 bg-gradient-to-r from-purple-500/10 to-blue-500/10 rounded-lg scale-0 group-hover:scale-100 transition-transform duration-300"></div>
                <div className="absolute bottom-0 left-1/2 w-0 h-0.5 bg-gradient-to-r from-purple-500 to-blue-500 group-hover:w-full group-hover:left-0 transition-all duration-300"></div>
              </Link>
            ))}
          </nav>

          <div className="hidden md:flex items-center space-x-4">
            <div className="flex items-center space-x-4 text-sm">
              <a
                href="tel:+2348012345678"
                className="flex items-center space-x-2 text-slate-600 hover:text-purple-600 transition-colors duration-300 group"
              >
                <div className="p-2 bg-gradient-to-r from-purple-500/10 to-blue-500/10 rounded-full group-hover:from-purple-500/20 group-hover:to-blue-500/20 transition-all duration-300">
                  <Phone className="h-4 w-4" />
                </div>
                <span className="hidden xl:block">+234 801 234 5678</span>
              </a>
              <a
                href="mailto:info@bishopknight.com"
                className="flex items-center space-x-2 text-slate-600 hover:text-purple-600 transition-colors duration-300 group"
              >
                <div className="p-2 bg-gradient-to-r from-purple-500/10 to-blue-500/10 rounded-full group-hover:from-purple-500/20 group-hover:to-blue-500/20 transition-all duration-300">
                  <Mail className="h-4 w-4" />
                </div>
              </a>
              <Link
                href="/shop"
                className="flex items-center space-x-2 text-slate-600 hover:text-purple-600 transition-colors duration-300 group relative"
              >
                <div className="p-2 bg-gradient-to-r from-purple-500/10 to-blue-500/10 rounded-full group-hover:from-purple-500/20 group-hover:to-blue-500/20 transition-all duration-300">
                  <ShoppingBag className="h-4 w-4" />
                </div>
              </Link>
            </div>
            <Link href="/quote">
              <Button className="relative overflow-hidden bg-gradient-to-r from-purple-600 to-blue-600 hover:from-purple-700 hover:to-blue-700 text-white px-6 py-2 rounded-full transition-all duration-300 group">
                <span className="relative z-10">Get Quote</span>
                <div className="absolute inset-0 bg-gradient-to-r from-white/20 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
              </Button>
            </Link>
          </div>

          <button
            onClick={() => setIsMenuOpen(!isMenuOpen)}
            className="lg:hidden p-2 text-slate-700 hover:text-purple-600 transition-colors duration-300"
          >
            <div className="relative">{isMenuOpen ? <X className="h-6 w-6" /> : <Menu className="h-6 w-6" />}</div>
          </button>
        </div>

        {isMenuOpen && (
          <div className="lg:hidden bg-white/95 backdrop-blur-xl border-t border-purple-100 py-4 animate-in slide-in-from-top duration-300">
            <nav className="flex flex-col space-y-2">
              {[
                { href: "/", label: "Home" },
                { href: "/about", label: "About" },
                { href: "/services", label: "Services" },
                { href: "/shop", label: "Shop" },
                { href: "/leadership", label: "Leadership" },
                { href: "/contact", label: "Contact" },
              ].map((item) => (
                <Link
                  key={item.href}
                  href={item.href}
                  className="text-slate-700 hover:text-purple-600 px-4 py-3 rounded-lg hover:bg-gradient-to-r hover:from-purple-500/10 hover:to-blue-500/10 transition-all duration-300"
                  onClick={() => setIsMenuOpen(false)}
                >
                  {item.label}
                </Link>
              ))}
              <div className="px-4 pt-4 border-t border-purple-100 space-y-3">
                <div className="flex items-center justify-center space-x-4">
                  <a
                    href="tel:+2348012345678"
                    className="flex items-center space-x-2 text-slate-600 hover:text-purple-600 transition-colors duration-300"
                  >
                    <Phone className="h-4 w-4" />
                    <span>+234 801 234 5678</span>
                  </a>
                  <a
                    href="mailto:info@bishopknight.com"
                    className="flex items-center space-x-2 text-slate-600 hover:text-purple-600 transition-colors duration-300"
                  >
                    <Mail className="h-4 w-4" />
                    <span>Email Us</span>
                  </a>
                </div>
                <Link href="/quote" onClick={() => setIsMenuOpen(false)}>
                  <Button className="w-full bg-gradient-to-r from-purple-600 to-blue-600 hover:from-purple-700 hover:to-blue-700 text-white rounded-full">
                    Get Quote
                  </Button>
                </Link>
              </div>
            </nav>
          </div>
        )}
      </div>
    </header>
  )
}

export default Header
