"use client"

import type React from "react"
import { useState } from "react"
import { Calculator, Download, Mail, ContrastIcon as Compare, CheckCircle, AlertCircle, Copy } from "lucide-react"
import { Button } from "@/components/ui/button"

interface QuoteResult {
  provider: string
  premium: number
  coverage: string
  benefits: string[]
  rating: number
}

export function QuoteFormSection() {
  const [formData, setFormData] = useState({
    insuranceType: "",
    name: "",
    email: "",
    phone: "",
    company: "",
    coverage: "",
    details: "",
  })

  const [isCalculating, setIsCalculating] = useState(false)
  const [quoteResults, setQuoteResults] = useState<QuoteResult[]>([])
  const [quoteNumber, setQuoteNumber] = useState("")
  const [showResults, setShowResults] = useState(false)

  const calculateQuotes = async (data: typeof formData) => {
    setIsCalculating(true)

    // Simulate API call to insurance calculators
    await new Promise((resolve) => setTimeout(resolve, 3000))

    const baseAmount = Number.parseFloat(data.coverage.replace(/[₦,]/g, "")) || 1000000
    const premiumRates = getRealPremiumRates(data.insuranceType, data.details)

    const realResults: QuoteResult[] = [
      {
        provider: "Leadway Assurance",
        premium: Math.round(premiumRates.leadway),
        coverage: `₦${baseAmount.toLocaleString()}`,
        benefits: ["24/7 Claims Support", "Nationwide Coverage", "Online Policy Management"],
        rating: 4.5,
      },
      {
        provider: "AXA Mansard",
        premium: Math.round(premiumRates.axa),
        coverage: `₦${baseAmount.toLocaleString()}`,
        benefits: ["Premium Customer Service", "Fast Claims Processing", "Mobile App"],
        rating: 4.7,
      },
      {
        provider: "AIICO Insurance",
        premium: Math.round(premiumRates.aiico),
        coverage: `₦${baseAmount.toLocaleString()}`,
        benefits: ["Comprehensive Coverage", "Risk Assessment", "Expert Advice"],
        rating: 4.3,
      },
    ]

    setQuoteResults(realResults)
    setIsCalculating(false)
    setShowResults(true)
  }

  const getRealPremiumRates = (insuranceType: string, details: string) => {
    const baseAmount = Number.parseFloat(formData.coverage.replace(/[₦,]/g, "")) || 1000000
    let basePremium = 0
    let negotiationNote = ""
    
    // Real premium calculation based on your provided rates
    switch (insuranceType) {
      case "motor-third-party":
        // Fixed ₦15,000/year for all insurers (no differences)
        basePremium = 15000
        break
        
      case "motor-comprehensive":
        // 5% of the value of the vehicle
        basePremium = baseAmount * 0.05
        break
          
      case "property":
        // Fire & Special Perils
        if (details.toLowerCase().includes("commercial") || details.toLowerCase().includes("business")) {
          basePremium = baseAmount * 0.003 // 0.3% for commercial
        } else {
          basePremium = baseAmount * 0.0025 // 0.25% for non-commercial residence
        }
        break
          
      case "marine":
        // Marine insurance
        if (details.toLowerCase().includes("clause a") || details.toLowerCase().includes("comprehensive")) {
          basePremium = baseAmount * 0.004 // 0.4% for Marine Clause A
        } else {
          basePremium = baseAmount * 0.0025 // 0.25% for Marine Clause C
        }
        break
          
      case "professional":
        // Professional Indemnity
        basePremium = baseAmount * 0.0125 // Average of 1-1.5% = 1.25%
        break
          
      case "group":
        // Group Life Insurance
        basePremium = baseAmount * 0.006 // 6 per mill (N6,000 for every N1 million)
        break
          
      case "business":
        // Check specific business insurance types
        if (details.toLowerCase().includes("burglary")) {
          basePremium = baseAmount * 0.004 // 0.4% for Burglary
        } else if (details.toLowerCase().includes("git") || details.toLowerCase().includes("goods in transit")) {
          basePremium = baseAmount * 0.005 // 0.5% for GIT
        } else if (details.toLowerCase().includes("fidelity")) {
          basePremium = baseAmount * 0.02 // 2% for Fidelity Guaranty
        } else if (details.toLowerCase().includes("all risk")) {
          basePremium = baseAmount * 0.05 // 5% for All Risk
        } else if (details.toLowerCase().includes("plant")) {
          basePremium = baseAmount * 0.0125 // Average of 1-1.5% for Plant All Risk
        } else {
          basePremium = baseAmount * 0.01 // Default 1% for general business
        }
        break
          
      default:
        basePremium = baseAmount * 0.01 // Default 1%
    }
    
    // For Motor Third Party: Same rate for all insurers (no variations)
    if (insuranceType === "motor-third-party") {
      return {
        leadway: basePremium,  // Same ₦15,000
        axa: basePremium,      // Same ₦15,000
        aiico: basePremium,    // Same ₦15,000
        negotiationNote
      }
    }
    
    // For other insurance types: Add slight variations for different providers
    return {
      leadway: basePremium * 1.05, // 5% higher
      axa: basePremium * 0.95,     // 5% lower (most competitive)
      aiico: basePremium * 1.02,   // 2% higher
      negotiationNote
    }
  }

  const exportToPDF = async () => {
    const jsPDF = (await import("jspdf")).default

    const doc = new jsPDF()

    const primaryColor = [147, 51, 234]
    const secondaryColor = [59, 130, 246]
    const textColor = [31, 41, 55]

    doc.setFillColor(primaryColor[0], primaryColor[1], primaryColor[2])
    doc.rect(0, 0, 210, 40, "F")
    doc.setFillColor(secondaryColor[0], secondaryColor[1], secondaryColor[2])
    doc.rect(0, 30, 210, 10, "F")

    doc.setTextColor(255, 255, 255)
    doc.setFontSize(24)
    doc.setFont("helvetica", "bold")
    doc.text("Bishop & Knight Insurance Brokers", 20, 25)

    doc.setTextColor(textColor[0], textColor[1], textColor[2])
    doc.setFontSize(20)
    doc.setFont("helvetica", "bold")
    doc.text("Insurance Quote Report", 20, 60)

    doc.setFontSize(12)
    doc.setFont("helvetica", "normal")
    let yPos = 105
    doc.text(`Quote Number: ${quoteNumber}`, 20, yPos)
    doc.text(`Date: ${new Date().toLocaleDateString()}`, 20, yPos + 10)

    doc.setFontSize(16)
    doc.setFont("helvetica", "bold")
    doc.text("Client Information", 20, yPos + 30)

    doc.setFontSize(11)
    doc.setFont("helvetica", "normal")
    yPos += 50
    doc.text(`Name: ${formData.name}`, 20, yPos)
    doc.text(`Email: ${formData.email}`, 20, yPos + 10)
    doc.text(`Phone: ${formData.phone}`, 20, yPos + 20)
    doc.text(`Company: ${formData.company || "N/A"}`, 20, yPos + 30)
    doc.text(`Insurance Type: ${formData.insuranceType}`, 20, yPos + 40)
    doc.text(`Coverage Amount: ${formData.coverage}`, 20, yPos + 50)

    yPos = 185
    doc.setFontSize(16)
    doc.setFont("helvetica", "bold")
    doc.text("Quote Comparison Results", 20, yPos)

    doc.setFillColor(240, 240, 240)
    doc.rect(20, yPos - 5, 170, 10, "F")
    doc.setFontSize(10)
    doc.setFont("helvetica", "bold")
    doc.text("Provider", 25, yPos)
    doc.text("Premium (₦)", 80, yPos)
    doc.text("Coverage", 120, yPos)
    doc.text("Rating", 160, yPos)

    doc.setFont("helvetica", "normal")
    quoteResults.forEach((result, index) => {
      yPos += 15
      if (yPos > 270) {
        doc.addPage()
        yPos = 30
      }

      if (index % 2 === 0) {
        doc.setFillColor(248, 250, 252)
        doc.rect(20, yPos - 5, 170, 10, "F")
      }

      doc.text(result.provider, 25, yPos)
      doc.text(result.premium.toLocaleString(), 80, yPos)
      doc.text(result.coverage, 120, yPos)
      doc.text(result.rating.toString(), 160, yPos)
    })

    const pageCount = doc.getNumberOfPages()
    for (let i = 1; i <= pageCount; i++) {
      doc.setPage(i)
      doc.setFontSize(8)
      doc.setTextColor(128, 128, 128)
      doc.text("Bishop & Knight Insurance Brokers Limited", 20, 285)
      doc.text(`Page ${i} of ${pageCount}`, 170, 285)
      doc.text("This quote is valid for 30 days from the date of issue", 20, 290)
    }

    doc.save(`insurance-quote-${quoteNumber}.pdf`)
  }

  const sendFollowUpEmail = async () => {
    const emailContent = `
Dear ${formData.name},

Thank you for requesting an insurance quote from Bishop & Knight Insurance Brokers Limited.

Your Quote Details:
- Insurance Type: ${formData.insuranceType}
- Coverage Amount: ${formData.coverage}
- Quote Number: ${quoteNumber}

We have generated competitive quotes from multiple providers for your ${formData.insuranceType} insurance needs. Our team will review your specific requirements and contact you within 24 hours to discuss the best options available.

Best Regards,
Bishop & Knight Insurance Team
Phone: +234-8029993330 | +234-8168026038
Email: info@bishopknight.com
    `

    try {
      // Get admin settings from localStorage (if available)
      const adminSettings = localStorage.getItem('adminSettings')
      const parsedSettings = adminSettings ? JSON.parse(adminSettings) : null
      
      const response = await fetch('/api/send-email', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          to: formData.email,
          subject: `Your Insurance Quote - ${quoteNumber}`,
          content: emailContent,
          adminSettings: parsedSettings // Pass admin settings to API
        })
      })
      
      const result = await response.json()
      
      if (response.ok) {
        alert(`✅ Email sent successfully to ${formData.email}`)
      } else {
        alert(`❌ Failed to send email: ${result.error}`)
        console.error('Email error:', result)
      }
    } catch (error) {
      alert('❌ Error sending email. Please try again.')
      console.error('Email sending error:', error)
    }
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()

    const quote = `BK-${Date.now()}-${Math.random().toString(36).substr(2, 4).toUpperCase()}`
    setQuoteNumber(quote)

    await calculateQuotes(formData)

    await sendFollowUpEmail()

    console.log("Quote form submitted:", formData)
  }

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement>) => {
    setFormData({
      ...formData,
      [e.target.name]: e.target.value,
    })
  }

  const copyQuoteNumber = () => {
    navigator.clipboard.writeText(quoteNumber)
    alert("Quote number copied to clipboard!")
  }

  return (
    <section id="quote-form" className="py-20 bg-white">
      <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="text-center mb-16">
          <h2 className="text-4xl md:text-5xl font-bold bg-gradient-to-r from-purple-600 to-blue-600 bg-clip-text text-transparent mb-6">
            Get Real-Time Insurance Quotes
          </h2>
          <p className="text-xl text-gray-600">
            Compare quotes from multiple providers instantly with our advanced calculator
          </p>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
          <div className="bg-gradient-to-br from-white to-gray-50 rounded-3xl shadow-2xl p-8 border border-gray-100">
            <form onSubmit={handleSubmit} className="space-y-6">
              <div className="group">
                <label htmlFor="insuranceType" className="block text-sm font-semibold text-gray-700 mb-2">
                  Insurance Type *
                </label>
                <select
                  id="insuranceType"
                  name="insuranceType"
                  required
                  value={formData.insuranceType}
                  onChange={handleChange}
                  className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:border-purple-500 focus:outline-none transition-colors duration-300 group-hover:border-gray-300"
                >
                  <option value="">Select insurance type</option>
                  <option value="motor-third-party">Motor Insurance - Third Party</option>
                  <option value="motor-comprehensive">Motor Insurance - Comprehensive</option>
                  <option value="property">Fire & Special Perils</option>
                  <option value="marine-clause-a">Marine Insurance - Clause A</option>
                  <option value="marine-clause-c">Marine Insurance - Clause C</option>
                  <option value="professional">Professional Indemnity</option>
                  <option value="group">Group Life Insurance</option>
                  <option value="business">Business Insurance (Burglary/GIT/Fidelity/All Risk/Plant)</option>
                  <option value="health">Health & Life Insurance</option>
                  <option value="advisory">Risk Advisory Services</option>
                </select>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div className="group">
                  <label htmlFor="name" className="block text-sm font-semibold text-gray-700 mb-2">
                    Full Name *
                  </label>
                  <input
                    type="text"
                    id="name"
                    name="name"
                    required
                    value={formData.name}
                    onChange={handleChange}
                    className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:border-purple-500 focus:outline-none transition-colors duration-300 group-hover:border-gray-300"
                    placeholder="Enter your full name"
                  />
                </div>

                <div className="group">
                  <label htmlFor="email" className="block text-sm font-semibold text-gray-700 mb-2">
                    Email Address *
                  </label>
                  <input
                    type="email"
                    id="email"
                    name="email"
                    required
                    value={formData.email}
                    onChange={handleChange}
                    className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:border-purple-500 focus:outline-none transition-colors duration-300 group-hover:border-gray-300"
                    placeholder="Enter your email address"
                  />
                </div>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div className="group">
                  <label htmlFor="phone" className="block text-sm font-semibold text-gray-700 mb-2">
                    Phone Number *
                  </label>
                  <input
                    type="tel"
                    id="phone"
                    name="phone"
                    required
                    value={formData.phone}
                    onChange={handleChange}
                    className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:border-purple-500 focus:outline-none transition-colors duration-300 group-hover:border-gray-300"
                    placeholder="Enter your phone number"
                  />
                </div>

                <div className="group">
                  <label htmlFor="company" className="block text-sm font-semibold text-gray-700 mb-2">
                    Company (Optional)
                  </label>
                  <input
                    type="text"
                    id="company"
                    name="company"
                    value={formData.company}
                    onChange={handleChange}
                    className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:border-purple-500 focus:outline-none transition-colors duration-300 group-hover:border-gray-300"
                    placeholder="Enter your company name"
                  />
                </div>
              </div>

              <div className="group">
                <label htmlFor="coverage" className="block text-sm font-semibold text-gray-700 mb-2">
                  Coverage Amount/Value *
                </label>
                <input
                  type="text"
                  id="coverage"
                  name="coverage"
                  required
                  value={formData.coverage}
                  onChange={handleChange}
                  className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:border-purple-500 focus:outline-none transition-colors duration-300 group-hover:border-gray-300"
                  placeholder="e.g., ₦5,000,000 or Vehicle value"
                />
              </div>

              <div className="group">
                <label htmlFor="details" className="block text-sm font-semibold text-gray-700 mb-2">
                  Additional Details *
                </label>
                <textarea
                  id="details"
                  name="details"
                  required
                  rows={4}
                  value={formData.details}
                  onChange={handleChange}
                  className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:border-purple-500 focus:outline-none transition-colors duration-300 group-hover:border-gray-300 resize-none"
                  placeholder="Please provide specific details about what you want to insure..."
                ></textarea>
              </div>

              <div className="text-center">
                <button
                  type="submit"
                  disabled={isCalculating}
                  className="group relative px-8 py-4 bg-gradient-to-r from-purple-500 to-blue-600 text-white font-semibold rounded-full hover:shadow-2xl hover:shadow-purple-500/25 transition-all duration-300 transform hover:scale-105 disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  <span className="relative z-10 flex items-center gap-2">
                    <Calculator className={`w-5 h-5 ${isCalculating ? "animate-spin" : ""}`} />
                    {isCalculating ? "Calculating Quotes..." : "Get Real-Time Quotes"}
                  </span>
                  <div className="absolute inset-0 bg-gradient-to-r from-purple-400 to-blue-500 rounded-full opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                </button>
              </div>
            </form>
          </div>

          <div className="space-y-6">
            {showResults && (
              <>
                <div className="bg-gradient-to-r from-green-50 to-emerald-50 rounded-2xl p-6 border border-green-200">
                  <div className="flex items-center gap-3 mb-4">
                    <CheckCircle className="w-6 h-6 text-green-600" />
                    <h3 className="text-lg font-bold text-green-800">Quote Generated Successfully!</h3>
                  </div>
                  <div className="bg-white rounded-xl p-4 border border-green-200">
                    <div className="flex items-center justify-between">
                      <div>
                        <p className="text-sm text-gray-600">Quote Number</p>
                        <p className="text-lg font-mono font-bold text-gray-900">{quoteNumber}</p>
                      </div>
                      <Button
                        onClick={copyQuoteNumber}
                        variant="outline"
                        size="sm"
                        className="flex items-center gap-2 bg-transparent"
                      >
                        <Copy className="w-4 h-4" />
                        Copy
                      </Button>
                    </div>
                    <div className="mt-3 p-3 bg-yellow-50 rounded-lg border border-yellow-200">
                      <div className="flex items-start gap-2">
                        <AlertCircle className="w-5 h-5 text-yellow-600 mt-0.5" />
                        <p className="text-sm text-yellow-800">
                          <strong>Important:</strong> Please keep this quote number in a safe place as it may be
                          required for support purposes.
                        </p>
                      </div>
                    </div>
                  </div>
                </div>

                <div className="bg-white rounded-2xl shadow-xl p-6 border border-gray-100">
                  <div className="flex items-center justify-between mb-6">
                    <h3 className="text-xl font-bold text-gray-900 flex items-center gap-2">
                      <Compare className="w-6 h-6 text-purple-600" />
                      Quote Comparison
                    </h3>
                    <div className="flex gap-2">
                      <Button
                        onClick={exportToPDF}
                        variant="outline"
                        size="sm"
                        className="flex items-center gap-2 bg-transparent"
                      >
                        <Download className="w-4 h-4" />
                        Export PDF
                      </Button>
                      <Button
                        onClick={sendFollowUpEmail}
                        variant="outline"
                        size="sm"
                        className="flex items-center gap-2 bg-transparent"
                      >
                        <Mail className="w-4 h-4" />
                        Email Results
                      </Button>
                    </div>
                  </div>

                  <div className="space-y-4">
                    {quoteResults.map((result, index) => (
                      <div
                        key={index}
                        className="border border-gray-200 rounded-xl p-4 hover:shadow-lg transition-shadow"
                      >
                        <div className="flex items-center justify-between mb-3">
                          <h4 className="font-bold text-lg text-gray-900">{result.provider}</h4>
                          <div className="flex items-center gap-1">
                            {[...Array(5)].map((_, i) => (
                              <span
                                key={i}
                                className={`text-sm ${i < Math.floor(result.rating) ? "text-yellow-400" : "text-gray-300"}`}
                              >
                                ★
                              </span>
                            ))}
                            <span className="text-sm text-gray-600 ml-1">({result.rating})</span>
                          </div>
                        </div>
                        <div className="grid grid-cols-2 gap-4 mb-3">
                          <div>
                            <p className="text-sm text-gray-600">Annual Premium</p>
                            <p className="text-2xl font-bold text-purple-600">₦{result.premium.toLocaleString()}</p>
                          </div>
                          <div>
                            <p className="text-sm text-gray-600">Coverage</p>
                            <p className="text-lg font-semibold text-gray-900">{result.coverage}</p>
                          </div>
                        </div>
                        <div>
                          <p className="text-sm text-gray-600 mb-2">Key Benefits</p>
                          <div className="flex flex-wrap gap-2">
                            {result.benefits.map((benefit, i) => (
                              <span key={i} className="px-3 py-1 bg-purple-100 text-purple-700 text-xs rounded-full">
                                {benefit}
                              </span>
                            ))}
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              </>
            )}

            {!showResults && (
              <div className="bg-gradient-to-br from-purple-50 to-blue-50 rounded-2xl p-8 text-center border border-purple-100">
                <Calculator className="w-16 h-16 text-purple-400 mx-auto mb-4" />
                <h3 className="text-xl font-bold text-gray-900 mb-2">Real-Time Quote Calculator</h3>
                <p className="text-gray-600">
                  Fill out the form to get instant quotes from multiple insurance providers with competitive rates and
                  comprehensive coverage options.
                </p>
              </div>
            )}
          </div>
        </div>
      </div>
    </section>
  )
}
