"use client"

import { Card, CardContent } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Users, Coffee, Lightbulb, Rocket, Heart } from "lucide-react"
import { useEffect, useState } from "react"

const cultureHighlights = [
  {
    icon: Lightbulb,
    title: "Innovation First",
    description: "We encourage creative thinking and bold ideas that push the boundaries of traditional insurance.",
    gradient: "from-yellow-500 to-orange-500",
    bg: "from-yellow-50 to-orange-50",
  },
  {
    icon: Users,
    title: "Collaborative Spirit",
    description: "Cross-functional teams working together to solve complex challenges and deliver exceptional results.",
    gradient: "from-blue-500 to-cyan-500",
    bg: "from-blue-50 to-cyan-50",
  },
  {
    icon: Rocket,
    title: "Growth Mindset",
    description:
      "Continuous learning and development opportunities to help our team members reach their full potential.",
    gradient: "from-purple-500 to-pink-500",
    bg: "from-purple-50 to-pink-50",
  },
  {
    icon: Heart,
    title: "Work-Life Balance",
    description: "Flexible work arrangements and wellness programs that prioritize our team's health and happiness.",
    gradient: "from-green-500 to-teal-500",
    bg: "from-green-50 to-teal-50",
  },
]

export function TeamCultureSection() {
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 },
    )

    const element = document.getElementById("team-culture")
    if (element) observer.observe(element)

    return () => observer.disconnect()
  }, [])

  return (
    <section
      id="team-culture"
      className="py-20 bg-gradient-to-br from-white via-purple-50/20 to-blue-50/20 relative overflow-hidden"
    >
      {/* Background Elements */}
      <div className="absolute inset-0 overflow-hidden">
        <div className="absolute top-32 right-20 w-80 h-80 bg-gradient-to-r from-purple-300/20 to-blue-300/20 rounded-full blur-3xl animate-blob"></div>
        <div className="absolute bottom-32 left-20 w-72 h-72 bg-gradient-to-r from-cyan-300/20 to-teal-300/20 rounded-full blur-3xl animate-blob animation-delay-2000"></div>
      </div>

      <div className="container mx-auto px-4 relative z-10">
        <div
          className={`text-center mb-16 transition-all duration-1000 ${isVisible ? "opacity-100 translate-y-0" : "opacity-0 translate-y-10"}`}
        >
          <div className="inline-flex items-center px-6 py-3 bg-gradient-to-r from-purple-100 to-blue-100 text-purple-800 rounded-full text-sm font-medium border border-purple-200/50 mb-6">
            <Coffee className="h-4 w-4 mr-2 animate-pulse" />
            Our Culture
          </div>

          <h2 className="text-4xl lg:text-5xl font-bold mb-6">
            <span className="bg-gradient-to-r from-purple-600 to-blue-600 bg-clip-text text-transparent">
              Where Talent
            </span>
            <br />
            <span className="text-slate-900">Meets Purpose</span>
          </h2>

          <p className="text-xl text-slate-600 max-w-3xl mx-auto leading-relaxed">
            Our culture is built on empowerment, innovation, and shared success. We believe that when our team thrives,
            our clients receive exceptional service.
          </p>
        </div>

        <div className="grid lg:grid-cols-2 gap-12 items-center mb-16">
          {/* Culture Image */}
          <div
            className={`relative transition-all duration-1000 ${isVisible ? "opacity-100 translate-x-0" : "opacity-0 -translate-x-10"}`}
          >
            <div className="relative overflow-hidden rounded-3xl shadow-2xl">
              <img
                src="https://images.unsplash.com/photo-1522202176988-66273c2fd55f?w=600&h=500&fit=crop&crop=center"
                alt="Bishop & Knight team collaboration and culture"
                className="w-full h-auto object-cover"
              />
              <div className="absolute inset-0 bg-gradient-to-t from-purple-900/30 to-transparent"></div>
            </div>

            {/* Floating Culture Stats */}
            <div className="absolute -top-6 -right-6 bg-white/90 backdrop-blur-xl p-4 rounded-xl shadow-xl animate-float">
              <div className="text-center">
                <div className="text-2xl font-bold text-purple-600">98%</div>
                <div className="text-xs text-slate-600">Team Satisfaction</div>
              </div>
            </div>

            <div className="absolute -bottom-6 -left-6 bg-white/90 backdrop-blur-xl p-4 rounded-xl shadow-xl animate-float animation-delay-1000">
              <div className="text-center">
                <div className="text-2xl font-bold text-blue-600">50+</div>
                <div className="text-xs text-slate-600">Team Members</div>
              </div>
            </div>
          </div>

          {/* Culture Content */}
          <div
            className={`space-y-8 transition-all duration-1000 delay-300 ${isVisible ? "opacity-100 translate-x-0" : "opacity-0 translate-x-10"}`}
          >
            <div className="space-y-6">
              <h3 className="text-3xl font-bold text-slate-900">Building Tomorrow's Insurance Leaders</h3>
              <p className="text-lg text-slate-600 leading-relaxed">
                At Bishop & Knight, we're not just building an insurance company – we're cultivating the next generation
                of industry innovators. Our culture emphasizes continuous learning, creative problem-solving, and
                meaningful impact.
              </p>
            </div>

            <div className="space-y-4">
              {[
                "Flexible remote and hybrid work options",
                "Professional development and certification programs",
                "Innovation time for personal projects",
                "Comprehensive health and wellness benefits",
                "Performance-based growth opportunities",
              ].map((benefit, index) => (
                <div key={index} className="flex items-center space-x-3">
                  <div className="w-2 h-2 bg-gradient-to-r from-purple-500 to-blue-500 rounded-full animate-pulse"></div>
                  <span className="text-slate-700">{benefit}</span>
                </div>
              ))}
            </div>

            <Button className="bg-gradient-to-r from-purple-600 to-blue-600 hover:from-purple-700 hover:to-blue-700 text-white px-8 py-3 rounded-full group transition-all duration-300">
              <span className="flex items-center">
                <Users className="mr-2 h-5 w-5 group-hover:scale-110 transition-transform duration-300" />
                Join Our Team
              </span>
            </Button>
          </div>
        </div>

        {/* Culture Highlights */}
        <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
          {cultureHighlights.map((highlight, index) => (
            <Card
              key={index}
              className={`group hover:shadow-2xl transition-all duration-500 border-0 shadow-lg hover:-translate-y-3 bg-gradient-to-br ${highlight.bg} backdrop-blur-sm overflow-hidden ${
                isVisible ? "opacity-100 translate-y-0" : "opacity-0 translate-y-10"
              }`}
              style={{ transitionDelay: `${(index + 4) * 150}ms` }}
            >
              <CardContent className="p-6 text-center relative">
                <div
                  className={`absolute inset-0 bg-gradient-to-r ${highlight.gradient} opacity-0 group-hover:opacity-5 transition-opacity duration-300`}
                ></div>
                <div className="relative z-10">
                  <div
                    className={`inline-flex items-center justify-center w-14 h-14 rounded-xl bg-gradient-to-r ${highlight.gradient} mb-4 group-hover:scale-110 transition-transform duration-300 shadow-lg`}
                  >
                    <highlight.icon className="h-7 w-7 text-white" />
                  </div>
                  <h4 className="text-lg font-bold text-slate-900 mb-3 group-hover:text-purple-700 transition-colors duration-300">
                    {highlight.title}
                  </h4>
                  <p className="text-slate-600 text-sm leading-relaxed">{highlight.description}</p>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    </section>
  )
}
